--[[
  Pano professional plugin for darktable 2.2.X

  copyright (c) 2017  Holger Klemm
  
  darktable is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.
  
  darktable is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with darktable.  If not, see <http://www.gnu.org/licenses/>.
]]

--[[
Version 2.1.1 for darktable 2.2.X

Depends:
- pto_gen
- cpfind
- cpclean
- linefind
- autooptimiser
- pano_modify
- nona
- enblend 4.2
- exiftool
   
Known bugs:   
- nona doesn't work with 32bit TIFF inputfiles. please use the 16bit or 8bit TIFF image export

Change report:
- fix path conflict with enfuse_pro 
- fix error, when exiftag and add to db is disabled 
- fix exiftool whitespace bug
- add no crop feature
- add french translation
- enhanced error message
   

Info workflow:
pto_gen -o project.pto *.jpg
cpfind -o project.pto --multirow --celeste project.pto
cpclean -o project.pto project.pto
linefind -o project.pto project.pto
autooptimiser -a -m -l -s -o project.pto project.pto
pano_modify --canvas=AUTO --crop=AUTO -o project.pto project.pto
nona -m TIFF_m -o project project.pto 
enblend -o project.tif project0000.tif project0001.tif project0002.tif   
exiftool
   
ADDITIONAL SOFTWARE NEEDED FOR THIS SCRIPT


USAGE
* require this file from your main luarc config file.

This plugin will add the new export modul "panorama image".
]]

local dt = require "darktable"
local gettext = dt.gettext

-- works with LUA API version 4.0.0
dt.configuration.check_version(...,{4,0,0},{5,0,0})

-- Tell gettext where to find the .mo file translating messages for a particular domain
gettext.bindtextdomain("pano_pro",dt.configuration.config_dir.."/lua/")

local function _(msgid)
    return gettext.dgettext("pano_pro", msgid)
end


-- command variables
homepath=os.getenv("HOME")
tmppath=homepath.."/.local/tmp"
images_to_process =""
first_imagefile =""
last_imagefile=""
counted_images=0
cmd_output_path=""
cmd_output_image_index=0
path_with_filename=""

-- pto_gen
cmd_pto_gen_projection_type=""
cmd_pto_gen_distortion=""
cmd_pto_gen_vignetting=""
cmd_pto_gen_fov=""
cmd_pto_gen_output=""
pto_genStartCommand=""
-- cpfind
cmd_cpfind_matching=""
cmd_cpfind_celeste=""
cmd_cpfind_celeste_radius=""
cmd_cpfind_celeste_threshold=""
cmd_cpfind_ransacmode=""
cmd_cpfind_ransaciter=""
cmd_cpfind_ransacdist=""
cmd_cpfind_cache=""
cmd_cpfind_fullscale=""
cmd_cpfind_input=""
cmd_cpfind_output=""
cpfindStartCommand=""
-- cpclean
cmd_cpclean_optimise_whole_panorama=""
cmd_cpclean_input=""
cmd_cpclean_output=""
cpcleanStartCommand=""
-- linefind
cmd_linefind_input=""
cmd_linefind_output=""
linefindStartCommand=""
--autooptimiser
cmd_autooptimiser_auto_align_mode=""
cmd_autooptimiser_pairwise_optimisation=""
cmd_autooptimiser_photometric_parameters=""
cmd_autooptimiser_level_horizon=""
cmd_autooptimiser_auto_projection=""
cmd_autooptimiser_input=""
cmd_autooptimiser_output=""
autooptimiserStartCommand=""
--pano_modify
cmd_pano_modify_canvas=""
cmd_pano_modify_crop=""
cmd_pano_modify_straighten=""
cmd_pano_modify_center=""
cmd_pano_modify_exposure=""
cmd_pano_modify_input=""
cmd_pano_modify_output=""
cmd_pano_modify_output_format=""
cmd_pano_modify_output_compression=""
pano_modifyStartCommand=""
--nona
cmd_nona_exposure=""
cmd_nona_output_format=""
cmd_nona_output_depth=""
cmd_nona_output_compression=""
cmd_nona_use_gpu=""
cmd_nona_prj_input=""
cmd_nona_bitmap_output=""
nonaStartCommand=""
--enblend
cmd_enblend_output_format=""
cmd_enblend_output_depth=""
cmd_enblend_output_compression=""
cmd_enblend_use_gpu=""
cmd_enblend_bitmap_input=""
cmd_enblend_bitmap_output=""
enblendStartCommand=""

-- initialize combobox and checkbox value for default
if (dt.preferences.read("pano_pro",  "selected_projection_type", "integer") == 0) then
-- pto_gen defaults
    dt.print_error("initialize combobox and checkbox value for default")
    dt.preferences.write("pano_pro", "selected_projection_type", "integer", 3) 
    dt.preferences.write("pano_pro", "checked_distortion", "bool", true)
    dt.preferences.write("pano_pro", "checked_vignetting", "bool", true)
    dt.preferences.write("pano_pro", "checked_fov", "bool", false)
    dt.preferences.write("pano_pro", "set_fov", "float", 50.0)
    dt.preferences.write("pano_pro", "visible_fov", "bool", false)
    dt.preferences.write("pano_pro", "selected_matching_strategy", "integer", 1) 
    dt.preferences.write("pano_pro", "checked_celeste", "bool", true)    
    dt.preferences.write("pano_pro", "selected_celeste_radius", "integer", 11)
    dt.preferences.write("pano_pro", "visible_celeste_radius", "bool", true)    
    dt.preferences.write("pano_pro", "selected_celeste_threshold", "integer", 5)
    dt.preferences.write("pano_pro", "visible_celeste_threshold", "bool", true)  
    dt.preferences.write("pano_pro", "checked_fullscale", "bool", false)
    dt.preferences.write("pano_pro", "checked_whole_pano_checking", "bool", true)  
    dt.preferences.write("pano_pro", "selected_ransacmode", "integer", 1) 
    dt.preferences.write("pano_pro", "selected_ransac_iterations", "integer", 2)
    dt.preferences.write("pano_pro", "selected_ransac_distance", "integer", 11)
    dt.preferences.write("pano_pro", "checked_auto_align_mode", "bool", true) 
    dt.preferences.write("pano_pro", "checked_pairwise_optimisation", "bool", false) 
    dt.preferences.write("pano_pro", "checked_optimise_photometric_parameters", "bool", true) 
    dt.preferences.write("pano_pro", "checked_level_horizon", "bool", true) 
    dt.preferences.write("pano_pro", "checked_auto_projection_size", "bool", true) 
    dt.preferences.write("pano_pro", "selected_canvas_size", "integer", 1)
    dt.preferences.write("pano_pro", "selected_output_crop", "integer", 1)
    dt.preferences.write("pano_pro", "checked_straighten", "bool", false)
    dt.preferences.write("pano_pro", "checked_center", "bool", true)
    dt.preferences.write("pano_pro", "selected_exposure", "integer", 1)
    dt.preferences.write("pano_pro", "set_exposure", "float", 2.0)
    dt.preferences.write("pano_pro", "visible_exposure", "bool", false)
    dt.preferences.write("pano_pro", "selected_output_format", "integer", 2)
    dt.preferences.write("pano_pro", "selected_overwrite", "integer", 1)  
end



local label_project_options= dt.new_widget("label")
{
     label = _('project options'),
     ellipsize = "start",
     halign = "end"
}

local label_line1= dt.new_widget("label")
{
     label = "_______________________________________________",
     ellipsize = "start",
     halign = "end"
}


combobox_projection_type = dt.new_widget("combobox")
{
    label = _('projection type'), 
    tooltip =_('Sets the output projection. For more informations have a look at:\nhttp://wiki.panotools.org/Projections'),
    value = dt.preferences.read("pano_pro", "selected_projection_type", "integer"), --0
    changed_callback = function(sel_projection_type) 
    dt.preferences.write("pano_pro", "selected_projection_type", "integer", sel_projection_type.selected)
    end,
    "rectilinear", "panorama (circular)","equirectangular","fisheye_ff", "stereographic","mercator","trans_mercator","sinusoidal","lambert_equal_area_conic","lambert_azimuthal","albers_equal_conic","miller_cylindrical","panini","architectural","orthographic","equisolid","equi_panini","biplane","triplane","panini_general","thoby","hammer",
  
    reset_callback = function(self_projection_type)
       self_projection_type.value = 3
    end
} 


check_button_distortion = dt.new_widget("check_button")
{
    label = _('distortion correction'), 
    value = dt.preferences.read("pano_pro", "checked_distortion", "bool"),
    tooltip =_('try to load distortion from database'), 
    clicked_callback = function(distortion)   
        if (distortion.value) then
           dt.preferences.write("pano_pro", "checked_distortion", "bool", true)
        else
           dt.preferences.write("pano_pro", "checked_distortion", "bool", false)
        end
    end,
    reset_callback = function(self_button_distortion)
       self_button_distortion.value = true
       dt.preferences.write("pano_pro", "checked_distortion", "bool", true)
    end
}


check_button_vignetting = dt.new_widget("check_button")
{
    label = _('vignetting correction'), 
    value = dt.preferences.read("pano_pro", "checked_vignetting", "bool"),
    tooltip =_('try to load vignetting from database'), 
    clicked_callback = function(vignetting)   
    if (vignetting.value) then
             dt.preferences.write("pano_pro", "checked_vignetting", "bool", true)
          else
             dt.preferences.write("pano_pro", "checked_vignetting", "bool", false)
          end
    end,
    reset_callback = function(self_button_vignetting)
       self_button_vignetting.value = true
       dt.preferences.write("pano_pro", "checked_vignetting", "bool", true)
    end
    
    
}


slider_fov = dt.new_widget("slider")
{
  label = _('horizontal field of view'),
  tooltip = _('Sets the horizontal field of view for all images. Useful if your lens does not store the focal length and/or crop factor correctly in the EXIF data.'),
  hard_min = 5,
  hard_max = 200,
  soft_min = 5,
  soft_max = 200,
  value = dt.preferences.read("pano_pro", "set_fov", "float"), --50 default
  sensitive= dt.preferences.read("pano_pro", "visible_fov", "bool"),
}

check_button_fov = dt.new_widget("check_button")
{
    label = _('horizontal field of view'), 
    value = dt.preferences.read("pano_pro", "checked_fov", "bool"),
    tooltip =_('Sets the horizontal field of view for all images. Useful if your lens does not store the focal length and/or crop factor correctly in the EXIF data.'), 
    clicked_callback = function(fov)   
      if (fov.value) then
        dt.preferences.write("pano_pro", "checked_fov", "bool", true)
        dt.preferences.write("pano_pro", "visible_fov", "bool", true)
        slider_fov.sensitive=true
      else
        dt.preferences.write("pano_pro", "checked_fov", "bool", false)
        dt.preferences.write("pano_pro", "visible_fov", "bool", false)
        slider_fov.sensitive=false
      end
    end,
    reset_callback = function(self_button_fov)
       self_button_fov.value = false
       dt.preferences.write("pano_pro", "checked_fov", "bool", false)
       dt.preferences.write("pano_pro", "visible_fov", "bool", false)
        slider_fov.sensitive=false
    end
}



local separator1 = dt.new_widget("separator")
{
}

local label_control_points= dt.new_widget("label")
{
     label = _('control point options'),
     ellipsize = "start",
     halign = "end"
}

local label_line2= dt.new_widget("label")
{
     label = "_______________________________________________",
     ellipsize = "start",
     halign = "end"
}


combobox_matching_strategy = dt.new_widget("combobox")
{
    label = _('matching strategy'), 
    tooltip =_("all pairs:\nthis is the default matching strategy. here all image pairs are matched against each other. E.g. if your project contains 5 images then cpfind matches the image pairs: 0-1, 0-2, 0-3, 0-4, 1-2, 1-3, 1-4, 2-3, 2-4 and 3-4\n\nlinear match:\nthis will only detect matches between adjacent images, e.g. for the 5 image example it will matches images pairs 0-1, 1-2, 2-3 and 3-4\n\nmultirow matching:\nThe algorithm is the same as described in multi-row panorama. By integrating this algorithm intocpfind it is faster by using several cores of modern CPUs and don't caching the keypoints to disc (which is time consuming)."),
    value = dt.preferences.read("pano_pro", "selected_matching_strategy", "integer"), --0
    changed_callback = function(sel_projection_type) 
    end,
    "all pairs", "linear match", "multirow matching",             
    reset_callback = function(self_points_error)
       self_points_error.value = dt.preferences.read("pano_pro", "selected_matching_strategy", "integer")
    end
} 


combobox_celeste_radius = dt.new_widget("combobox")
{
    label = _('celeste radius'), 
    tooltip =_('the celeste step can be fine tuned by the parameters'),
    sensitive=dt.preferences.read("pano_pro", "visible_celeste_radius", "bool"),
    value = dt.preferences.read("pano_pro", "selected_celeste_radius", "integer"), --1
    changed_callback = function(sel_celeste_radius) 
    end,
    "10", "11", "12","13","14","15","16","17","18","19","20","21","22","23","24","25","26","27","28","29","30",             
    reset_callback = function(self_celeste_radius)
       self_celeste_radius.value = dt.preferences.read("pano_pro", "selected_celeste_radius", "integer")
    end
} 


combobox_celeste_threshold = dt.new_widget("combobox")
{
    label = _('celeste threshold'), 
    tooltip =_('the celeste step can be fine tuned by the parameters'),
    sensitive=dt.preferences.read("pano_pro", "visible_celeste_threshold", "bool"),
    value = dt.preferences.read("pano_pro", "selected_celeste_threshold", "integer"), --1
    changed_callback = function(sel_celeste_threshold) 
    end,
    "0.1", "0.2", "0.3","0.4","0.5","0.6","0.7","0.8","0.9","1.0",             
    reset_callback = function(self_celeste_threshold)
       self_celeste_threshold.value = dt.preferences.read("pano_pro", "selected_celeste_threshold", "integer")
    end
} 


check_button_celeste = dt.new_widget("check_button")
{
    label = _('celeste algorithm'), 
    value = dt.preferences.read("pano_pro", "checked_celeste", "bool"),
    tooltip =_('outdoor panorama often contains clouds. clouds are bad areas for setting control points because they are moving object. the celeste algorithm can masked out areas which contains clouds. this is only done internal for the keypoint finding step and does not change the alpha channel of your image.'), 
    clicked_callback = function(celeste)   
    if (celeste.value) then
             dt.preferences.write("pano_pro", "checked_celeste", "bool", true)
             dt.preferences.write("pano_pro", "visible_celeste_radius", "bool", true)
             dt.preferences.write("pano_pro", "visible_celeste_threshold", "bool", true)
             combobox_celeste_radius.sensitive=true
             combobox_celeste_threshold.sensitive=true
          else
             dt.preferences.write("pano_pro", "checked_celeste", "bool", false)
             dt.preferences.write("pano_pro", "visible_celeste_radius", "bool", false)
             dt.preferences.write("pano_pro", "visible_celeste_threshold", "bool", false)
             combobox_celeste_radius.sensitive=false
             combobox_celeste_threshold.sensitive=false
          end
    end,
    reset_callback = function(self_celeste)
       self_celeste.value = true
       combobox_celeste_radius.sensitive=true
       combobox_celeste_threshold.sensitive=true
       dt.preferences.write("pano_pro", "checked_celeste", "bool", true)
       dt.preferences.write("pano_pro", "visible_celeste_radius", "bool", true)
       dt.preferences.write("pano_pro", "visible_celeste_threshold", "bool", true)
    end
}




check_button_whole_pano_checking = dt.new_widget("check_button")
{
    label = _('pairwise optimisation from control points'), 
    value = dt.preferences.read("pano_pro", "checked_whole_pano_checking", "bool"),
    tooltip =_('optimises all images pairs, calculates for each pair mean and standard deviation and removes all control points with error bigger than mean+n*sigma. this option skips the optimisation for the whole panorama'), 
    clicked_callback = function(wholepanochecking)   
    if (wholepanochecking.value) then
             dt.preferences.write("pano_pro", "checked_whole_pano_checking", "bool", true)
          else
             dt.preferences.write("pano_pro", "checked_whole_pano_checking", "bool", false)
          end
    end,
    reset_callback = function(self_whole_pano_checking)
      self_whole_pano_checking.value=true
      dt.preferences.write("pano_pro", "checked_whole_pano_checking", "bool", true)
    end
}


check_button_fullscale= dt.new_widget("check_button")
{
    label = _('uses full scale image to detect keypoints'), 
    value = dt.preferences.read("pano_pro", "checked_fullscale", "bool"),
    tooltip =_('For speed reasons cpfind is using images, which are scaled to their half width and height, to find keypoints. With this option cpfind is working on the full scale images. This takes longer but can provide \"better\" and/or more control points.'), 
    clicked_callback = function(fullscale)   
    if (fullscale.value) then
             dt.preferences.write("pano_pro", "checked_fullscale", "bool", true)
          else
             dt.preferences.write("pano_pro", "checked_fullscale", "bool", false)
          end
    end,
    reset_callback = function(self_fullscale)
      self_fullscale.value=true
      dt.preferences.write("pano_pro", "checked_fullscale", "bool", true)
    end
}




combobox_ransacmode = dt.new_widget("combobox")
{
    label = _('random sample consensus mode'), 
    tooltip =_('auto:\nuse homography for images with hfov < 65 degrees and rpy otherwise.\n\nhom:\nAssume a homography. Only applicable for non-wide angle views. Uses the original panomatic code. It is also more flexible than required and can generate false matches, particularly if most of the matches are located on a single line.\n\nrpy:\nAlign images using roll, pitch and yaw. This requires a good estimate for the horizontal field of view (and distortion, for heavily distorted images). It is the preferred mode if a calibrated lens is used, or the HFOV could be read successfully from the EXIF data.\n\nrpyv:\nAlign pair by optimizing roll, pitch, yaw and field of view. Should work without prior knowledge of the field of view, but might fail more often, due to error function used in the panotools optimizer, it tends to shrink the fov to 0.\n\nrpyvb:\nAlign pair by optimizing roll, pitch, yaw, field of view and the "b" distortion parameter.  Probably very fragile, just implemented for testing.'),
    value = dt.preferences.read("pano_pro", "selected_ransacmode", "integer"), --1
    changed_callback = function(sel_ransacmode) 
    end,
    "auto", "hom", "rpy","rpyv","rpyb",             
    reset_callback = function(self_ransacmode)
       self_ransacmode.value = 1
       dt.preferences.write("pano_pro", "selected_ransacmode", "integer", 1) 
    end
} 

combobox_ransac_iterations = dt.new_widget("combobox")
{
    label = _('random sample consensus iterations'), 
    tooltip =_('ransac: iterations (default : 1000)'),
    value = dt.preferences.read("pano_pro", "selected_ransac_iterations", "integer"), --1
    changed_callback = function(sel_ransaciter) 
    end,
    "500", "1000", "1500","2000","2500",             
    reset_callback = function(self_sel_ransaciter)
       self_sel_ransaciter.value = 2
       dt.preferences.write("pano_pro", "selected_ransac_iterations", "integer", 2)
    end
} 

combobox_ransac_distance = dt.new_widget("combobox")
{
    label = _('random sample consensus distance'), 
    tooltip =_('ransac: homography estimation distance threshold (pixels) (default : 25)'),
    value = dt.preferences.read("pano_pro", "selected_ransac_distance", "integer"), --1
    changed_callback = function(sel_ransacdist) 
    end,
    "15", "16", "17","18","19","20","21","22","23","24","25","26","27","28","29","30","31","32","33","34","35",             
    reset_callback = function(self_ransacdis)
       self_ransacdis.value = 11
       dt.preferences.write("pano_pro", "selected_ransac_distance", "integer", 11)
    end
} 

local separator2 = dt.new_widget("separator")
{
}

local label_pano_options= dt.new_widget("label")
{
     label = _('panorama options'),
     ellipsize = "start",
     halign = "end"
}

local label_line3= dt.new_widget("label")
{
     label = "_______________________________________________",
     ellipsize = "start",
     halign = "end"
}


check_button_alignmode = dt.new_widget("check_button")
{
    label = _('auto align mode'), 
    value = dt.preferences.read("pano_pro", "checked_auto_align_mode", "bool"),
    tooltip =_('auto align mode, includes various optimisation stages, depending on the amount and distribution of the control points'), 
    clicked_callback = function(auto_align_mode)   
    if (auto_align_mode.value) then
             dt.preferences.write("pano_pro", "checked_auto_align_mode", "bool", true)
          else
             dt.preferences.write("pano_pro", "checked_auto_align_mode", "bool", false)
          end
    end,
    reset_callback = function(self_alignmode)
      self_alignmode.value=true
      dt.preferences.write("pano_pro", "checked_auto_align_mode", "bool", true)
    end
}


check_button_pairwise_optimisation = dt.new_widget("check_button")
{
    label = _('pairwise align optimisation'), 
    value = dt.preferences.read("pano_pro", "checked_pairwise_optimisation", "bool"),
    tooltip =_('auto align mode, includes various optimisation stages, depending on the amount and distribution of the control points'), 
    clicked_callback = function(pairwise_optimisation)   
    if (pairwise_optimisation.value) then
             dt.preferences.write("pano_pro", "checked_pairwise_optimisation", "bool", true)
          else
             dt.preferences.write("pano_pro", "checked_pairwise_optimisation", "bool", false)
          end
    end,
    reset_callback = function(self_pairwise_optimisation)
      self_pairwise_optimisation.value=false
      dt.preferences.write("pano_pro", "checked_pairwise_optimisation", "bool", false)
    end
}


check_button_optimise_photometric_parameters = dt.new_widget("check_button")
{
    label = _('optimise photometric parameters'), 
    value = dt.preferences.read("pano_pro", "checked_optimise_photometric_parameters", "bool"),
    tooltip =_('optimise photometric parameters'), 
    clicked_callback = function(optimise_photometric_parameters)   
    if (optimise_photometric_parameters.value) then
             dt.preferences.write("pano_pro", "checked_optimise_photometric_parameters", "bool", true)
          else
             dt.preferences.write("pano_pro", "checked_optimise_photometric_parameters", "bool", false)
          end
    end,
    reset_callback = function(self_optimise_photometric_parameters)
      self_optimise_photometric_parameters.value=true
      dt.preferences.write("pano_pro", "checked_optimise_photometric_parameters", "bool", true)
    end
}


check_button_level_horizon = dt.new_widget("check_button")
{
    label = _('level horizon'), 
    value = dt.preferences.read("pano_pro", "checked_level_horizon", "bool"),
    tooltip =_('Level horizon (works best for horizontal panoramas)'), 
    clicked_callback = function(level_horizon)   
    if (level_horizon.value) then
             dt.preferences.write("pano_pro", "checked_level_horizon", "bool", true)
          else
             dt.preferences.write("pano_pro", "checked_level_horizon", "bool", false)
          end
    end,
    reset_callback = function(self_level_horizon)
      self_level_horizon.value=true
      dt.preferences.write("pano_pro", "checked_level_horizon", "bool", true)
    end
}


check_button_auto_projection_size = dt.new_widget("check_button")
{
    label = _('auto output projection and size'), 
    value = dt.preferences.read("pano_pro", "checked_auto_projection_size", "bool"),
    tooltip =_('automatically select a suitable output projection and size'), 
    clicked_callback = function(auto_projection_size)   
    if (auto_projection_size.value) then
             dt.preferences.write("pano_pro", "checked_auto_projection_size", "bool", true)
          else
             dt.preferences.write("pano_pro", "checked_auto_projection_size", "bool", false)
          end
    end,
    reset_callback = function(self_auto_projection_size)
      self_auto_projection_size.value=true
      dt.preferences.write("pano_pro", "checked_auto_projection_size", "bool", true)
    end
}



combobox_canvas_size = dt.new_widget("combobox")
{
    label = _('output canvas size'),
    tooltip = _('sets the output canvas size'),
    value = dt.preferences.read("pano_pro", "selected_canvas_size", "integer"), 
    changed_callback = function(sel_canvas_size)
      dt.preferences.write("pano_pro", "selected_canvas_size", "integer", sel_canvas_size.selected)
    end,
    "auto","100%","95%","90%","85%","80%","75%","70%","65%","60%","55%","50%","45%","40%",
    reset_callback = function(self_canvas_size)
      self_canvas_size.value=1
      dt.preferences.write("pano_pro", "selected_canvas_size", "integer", 1)
    end
}


slider_exposure = dt.new_widget("slider")
{
  label = _('output exposure value'),
  tooltip = _('Set exposure for ldr mode'),
  hard_min = 0,
  hard_max = 5,
  soft_min = 0,
  soft_max = 5,
  value = dt.preferences.read("pano_pro", "set_exposure", "float"), --2.0 default
  sensitive = dt.preferences.read("pano_pro", "visible_exposure", "bool"), 
}


combobox_exposure = dt.new_widget("combobox")
{
    label = _('output exposure'),
    tooltip = _('sets the output exposure'),
    value = dt.preferences.read("pano_pro", "selected_exposure", "integer"), 
    changed_callback = function(sel_exposure)
      dt.preferences.write("pano_pro", "selected_exposure", "integer", sel_exposure.selected)
      if (sel_exposure.selected == 1) then
          slider_exposure.sensitive=false
          dt.preferences.write("pano_pro", "visible_exposure", "bool", false)
      else
          slider_exposure.sensitive=true
          dt.preferences.write("pano_pro", "visible_exposure", "bool", true)
      end
    end,
    "auto","value",
    reset_callback = function(self_exposure)
      self_exposure.value=1
      dt.preferences.write("pano_pro", "selected_exposure", "integer", 1)
    end
}





combobox_output_crop = dt.new_widget("combobox")
{
    label = _('output crop'),
    tooltip = _('Sets the crop rectangle.\n\nauto: autocrop panorama\nautohdr: autocrop HDR panorama'),
    value = dt.preferences.read("pano_pro", "selected_output_crop", "integer"), 
    changed_callback = function(sel_output_crop)
      dt.preferences.write("pano_pro", "selected_output_crop", "integer", sel_output_crop.selected)
    end,
    "auto", "autohdr","no",
    reset_callback = function(self_output_crop)
       self_output_crop.value = 1
       dt.preferences.write("pano_pro", "selected_output_crop", "integer", 1)
    end
}


check_button_straighten = dt.new_widget("check_button")
{
    label = _('straightens the panorama'), 
    value = dt.preferences.read("pano_pro", "checked_straighten", "bool"),
    tooltip =_('A common problem when creating a panorama is a curved horizon. This can occur if the panorama was taken with the camera pointing upwards or downwards. Use this option to straightens the panorama'), 
    clicked_callback = function(auto_straighten)   
    if (auto_straighten.value) then
             dt.preferences.write("pano_pro", "checked_straighten", "bool", true)
          else
             dt.preferences.write("pano_pro", "checked_straighten", "bool", false)
          end
    end,
    reset_callback = function(self_straighten)
       self_straighten.value = false
       dt.preferences.write("pano_pro", "checked_straighten", "bool", false)
    end
    
    
}


check_button_center = dt.new_widget("check_button")
{
    label = _('centers the panorama'), 
    value = dt.preferences.read("pano_pro", "checked_center", "bool"),
    tooltip =_('centers the panorama'), 
    clicked_callback = function(center)   
    if (center.value) then
             dt.preferences.write("pano_pro", "checked_center", "bool", true)
          else
             dt.preferences.write("pano_pro", "checked_center", "bool", false)
          end
    end,
    reset_callback = function(self_center)
       self_center.value = false
       dt.preferences.write("pano_pro", "checked_center", "bool", false)
    end
    
}



local separator3 = dt.new_widget("separator")
{
}


local label_output_format= dt.new_widget("label")
{
     label = _('target file'),
     ellipsize = "start",
     halign = "end"
}


local label_line4= dt.new_widget("label")
{
     label = "_______________________________________________",
     ellipsize = "start",
     halign = "end"
}

combobox_output_format = dt.new_widget("combobox")
{
    label = _('file format'),
    value = dt.preferences.read("pano_pro", "selected_output_format", "integer"), --2, "TIFF", "JPEG", "PNG","PNM","PBM","PGM","PPM",
    changed_callback = function(sel_output_format)
      dt.preferences.write("pano_pro", "selected_output_format", "integer", sel_output_format.selected)
    end,
    "TIFF", "JPEG", "PNG",
    reset_callback = function(self_output_format)
       self_output_format.value = dt.preferences.read("pano_pro", "selected_output_format", "integer")
    end
}


local label_path = dt.new_widget("label")
{
     label = _('directory'),
     ellipsize = "start",
     halign = "start"
}

file_chooser_button_path_pano = dt.new_widget("file_chooser_button")
{
    title = _('select export path'),  -- The title of the window when choosing a file
    is_directory = true,             -- True if the file chooser button only allows directories to be selecte
    tooltip =_('select the target directory for the panorama image. \nthe filename is created automatically.')
}

combobox_existing_file = dt.new_widget("combobox")
{
    label = _('on conflict'),
    value = dt.preferences.read("pano_pro", "selected_overwrite", "integer"), --1,
    changed_callback = function(sel_overwrite)
       dt.preferences.write("pano_pro", "selected_overwrite", "integer", sel_overwrite.selected)
    end,
    _('create unique filename'),_('overwrite'),
    reset_callback = function(self_overwrite)
       self_overwrite.value = 1
       dt.preferences.write("pano_pro", "selected_overwrite", "integer", 1)
    end
}








--FUNCTION  
  
local widget = dt.new_widget("box") {
    orientation = "vertical",
    --project options
    label_project_options,
    label_line1,
    combobox_projection_type,
    check_button_distortion,
    check_button_vignetting,
    check_button_fov,
    slider_fov,
    separator1,
    -- control point options
    label_control_points,
    label_line2,
    combobox_matching_strategy,
    check_button_celeste,
    combobox_celeste_radius,
    combobox_celeste_threshold,
    combobox_ransacmode,
    combobox_ransac_iterations,
    combobox_ransac_distance,
    check_button_fullscale,
    check_button_whole_pano_checking,
    -- panorama options
    separator2,
    label_pano_options,
    label_line3,
    check_button_alignmode,
    check_button_pairwise_optimisation,
    check_button_optimise_photometric_parameters,
    check_button_level_horizon,
    check_button_auto_projection_size,
    combobox_canvas_size,
    combobox_output_crop,
    check_button_straighten,
    check_button_center,
    combobox_exposure,
    slider_exposure,
    -- file options
    separator3,
    label_output_format,
    label_line4,
    combobox_output_format,
    label_path,
    file_chooser_button_path_pano,
    combobox_existing_file,
    
}




local function GetFileName(full_path)
      local filename_with_suffix=full_path:match("[^/]+$" )
      return string.sub(filename_with_suffix,1, -5)
end

local function truncate(x)
      return x<0 and math.ceil(x) or math.floor(x)
end

local function replace_comma_to_dot(s)
	return string.gsub(s, "%,", ".")
end

local function checkIfBinExists(bin)
  local handle = io.popen("which "..bin)
  local result = handle:read()
  local ret
  handle:close()
  if (result) then
    ret = true
  else
    ret = false
  end


  return ret
end

local function file_exists(name)
   local f=io.open(""..name.."","r")
   if f~=nil then
       io.close(f)
       return true
   else
       return false
   end
end

local function remove_tmp_files()
   dt.control.execute("cpfind --clean \""..tmppath.."/project.pto\"") 
   dt.control.execute("rm \""..tmppath.."/project.pto\"") 
   dt.control.execute("rm "..images_to_process) 
   dt.control.execute("rm "..cmd_enblend_bitmap_input)   
   counted_images=0
   images_to_process=""
end


local function stop_selection(job)
    job.valid = false
end



local function show_status(storage, image, format, filename,
  number, total, high_quality, extra_data)
     dt.print(_('export TIFF images for panorama ')..tostring(truncate(number)).." / "..tostring(truncate(total)))   
end

  



local function create_pano_image(storage, image_table, extra_data) --finalize
job = dt.gui.create_job(_('creating panorama image'), true, stop_selection)
    
-- delete old projects if exists
projectfile=tmppath.."/project.pto"
   if (file_exists(projectfile)) then
      dt.control.execute("rm \""..tmppath.."/project.pto\"") 
      dt.print_error("old project file project.pto removed")
   end

-- create image path and infos
counted_images=0
        for _,v in pairs(image_table) do
                 images_to_process = images_to_process.."\"" ..v.."\" "
                 counted_images=counted_images+1
                 if (counted_images == 1) then
                 first_imagefile=v  
                 else
                 last_imagefile=v    
                 end
                
        end
            if (counted_images<=1) then
                dt.print(_('ERROR: not enough pictures selected. please select two or more images\nfrom the panorama.'))
               -- os.remove(images_to_process)
                dt.control.execute("rm "..images_to_process)
                dt.print_error("rm "..images_to_process)
                return
            elseif (counted_images>=21) then  
                dt.print(_('you have selected more then 20 images. the panorama process could take a very long time! \nhave a nice beake.'))
            end   
            
-- check installed software
enblendVersionStartCommand='enblend --version | grep "enblend 4.2"'
enblend_version=dt.control.execute(enblendVersionStartCommand)

   if (not (checkIfBinExists("pto_gen"))) then
     dt.print(_('ERROR: pto_gen not found. please install hugin.'))
     dt.print_error(_('pto_gen not found. please install hugin.'))
     dt.control.execute("rm "..images_to_process)
     return
   elseif  (not (checkIfBinExists("cpfind"))) then
     dt.print(_('ERROR: cpfind not found. please install hugin.'))
     dt.print_error(_('cpfind not found. please install hugin.'))
     dt.control.execute("rm "..images_to_process)
     return
   elseif  (not (checkIfBinExists("cpclean"))) then
     dt.print(_('ERROR: cpclean not found. please install hugin.'))
     dt.print_error(_('cpclean not found. please install hugin.'))
     dt.control.execute("rm "..images_to_process)
     return  
   elseif  (not (checkIfBinExists("linefind"))) then
     dt.print(_('ERROR: linefind not found. please install hugin.'))
     dt.print_error(_('linefind not found. please install hugin.'))
     dt.control.execute("rm "..images_to_process)
     return    
   elseif  (not (checkIfBinExists("autooptimiser"))) then
     dt.print(_('ERROR: autooptimiser not found. please install hugin.'))
     dt.print_error(_('autooptimiser not found. please install hugin.'))
     dt.control.execute("rm "..images_to_process)
     return   
   elseif  (not (checkIfBinExists("pano_modify"))) then
     dt.print(_('ERROR: pano_modify not found. please install hugin.'))
     dt.print_error(_('pano_modify not found. please install hugin.'))
     dt.control.execute("rm "..images_to_process)
     return     
    elseif  (not (checkIfBinExists("nona"))) then
     dt.print(_('ERROR: nona not found. please install hugin.'))
     dt.print_error(_('nona not found. please install hugin.'))
     dt.control.execute("rm "..images_to_process)
     return  
    elseif (not (checkIfBinExists("enblend")) and (enblend_version ~= 0)) then 
     dt.print(_('ERROR: enblend 4.2 not found. please install enblend version 4.2.'))
     dt.print_error(_('enblend 4.2 not found. please install enblend version 4.2.'))
     dt.control.execute("rm "..images_to_process)
    elseif  (not (checkIfBinExists("exiftool"))) then
     dt.print(_('ERROR: exiftool not found. please install exiftool.'))
     dt.print_error(_('exiftool not found. please install exiftool.'))
     dt.control.execute("rm "..images_to_process)
     return   
   end 
job.percent = 0.1   
-- check path    
    cmd_output_path = file_chooser_button_path_pano.value
    if (cmd_output_path == nil) then
       dt.print(_('ERROR: no target directory selected'))  
       return
    else    
        
-- main program
      -- create and execute pto_gen command 
      ptype=tonumber(combobox_projection_type.selected) -1 
      cmd_pto_gen_projection_type="-p "..tostring(ptype)
      if (check_button_distortion.value) then
         cmd_pto_gen_distortion="--distortion"
      else
         cmd_pto_gen_distortion=""
      end
      
      if (check_button_vignetting.value) then
         cmd_pto_gen_vignetting="--vignetting"   
      else
         cmd_pto_gen_vignetting=""
      end
      
      if (check_button_fov.value) then
         cmd_pto_gen_fov="-f "..slider_fov.value 
      else
         cmd_pto_gen_fov=""
      end
      cmd_pto_gen_output="-o \""..tmppath.."/project.pto\""
      
      pto_genStartCommand="pto_gen "..cmd_pto_gen_projection_type.." "..cmd_pto_gen_distortion.." "..cmd_pto_gen_vignetting.." "..cmd_pto_gen_fov.." "..cmd_pto_gen_output.." "..images_to_process
      dt.print_error(pto_genStartCommand)
      dt.print(_('creating panorama project'))
      result_pto_gen=dt.control.execute(pto_genStartCommand)    
job.percent = 0.2
      if (result_pto_gen == 0) then
      -- create and execute cpfind command  
           if (combobox_matching_strategy.selected == 1) then
              cmd_cpfind_matching=""  
           elseif (combobox_matching_strategy.selected == 2) then
              cmd_cpfind_matching="--linearmatch" 
           elseif (combobox_matching_strategy.selected == 2) then
              cmd_cpfind_matching="--multirow" 
           else
             cmd_cpfind_matching=""  
           end
           
           if (check_button_celeste.value) then
              cmd_cpfind_celeste="--celeste"
              cmd_cpfind_celeste_radius="--celesteradius "..combobox_celeste_radius.value
              cmd_cpfind_celeste_threshold="--celestethreshold "..combobox_celeste_threshold.value
           else
              cmd_cpfind_celeste=""
              cmd_cpfind_celeste_radius=""
              cmd_cpfind_celeste_threshold=""  
           end
           
           cmd_cpfind_ransacmode="--ransacmode "..combobox_ransacmode.value
           cmd_cpfind_ransaciter="--ransaciter "..combobox_ransac_iterations.value
           cmd_cpfind_ransacdist="--ransacdist "..combobox_ransac_distance.value
       
           if (dt.preferences.read("pano_pro", "keypoint_cache", "bool")) then   
           cmd_cpfind_cache="--cache"
           else
           cmd_cpfind_cache=""
           end
           cmd_cpfind_input="\""..tmppath.."/project.pto\""
           cmd_cpfind_output="-o \""..tmppath.."/project.pto\""       
           if (check_button_fullscale.value) then
           cmd_cpfind_fullscale="--fullscale"
           else    
           cmd_cpfind_fullscale=""   
           end
           
           cpfindStartCommand="cpfind "..cmd_cpfind_matching.." "..cmd_cpfind_celeste.." "..cmd_cpfind_celeste_radius.." "..cmd_cpfind_celeste_threshold.." "..cmd_cpfind_ransacmode.." "..cmd_cpfind_ransaciter.." "..cmd_cpfind_ransacdist.." "..cmd_cpfind_cache.." "..cmd_cpfind_fullscale.." "..cmd_cpfind_output.." "..cmd_cpfind_input
           dt.print_error(cpfindStartCommand)
           dt.print(_('creating control points'))
           result_cpfind=dt.control.execute(cpfindStartCommand) 
job.percent = 0.4
           if (result_cpfind == 0) then
               
                    -- create cpclean command  
            if (check_button_whole_pano_checking.value) then
               cmd_cpclean_optimise_whole_panorama="-p"
            else
               cmd_cpclean_optimise_whole_panorama=""  
            end              
            cmd_cpclean_input="\""..tmppath.."/project.pto\""
            cmd_cpclean_output="-o \""..tmppath.."/project.pto\""
            cpcleanStartCommand="cpclean "..cmd_cpclean_optimise_whole_panorama.." "..cmd_cpclean_output.." "..cmd_cpclean_input
            dt.print_error(cpcleanStartCommand)
            result_cpclean=dt.control.execute(cpcleanStartCommand) 
            dt.print_error("CPCLEAN RETURN "..result_cpclean)
job.percent = 0.5
               if (result_cpclean == 0) then
                    -- create linefind command     
                    cmd_linefind_input="\""..tmppath.."/project.pto\""
                    cmd_linefind_output="-o \""..tmppath.."/project.pto\""
                    linefindStartCommand="linefind "..cmd_linefind_output.." "..cmd_linefind_input
                    dt.print_error(linefindStartCommand)
                    dt.print(_('searching lines'))
                    result_linefind=dt.control.execute(linefindStartCommand) 
                    dt.print_error("LINEFIND RETURM "..result_linefind)
job.percent = 0.6
                    if (result_linefind == 0) then
                    -- create autooptimiser command   
                       if (check_button_alignmode.value) then
                          cmd_autooptimiser_auto_align_mode="-a" 
                       else
                          cmd_autooptimiser_auto_align_mode=""
                       end
                       if (check_button_pairwise_optimisation.value) then
                           cmd_autooptimiser_pairwise_optimisation="-p"
                       else
                           cmd_autooptimiser_pairwise_optimisation=""
                       end
                       if (check_button_optimise_photometric_parameters.value) then
                          cmd_autooptimiser_photometric_parameters="-m"
                       else
                          cmd_autooptimiser_photometric_parameters=""
                       end
                       if (check_button_level_horizon.value) then
                          cmd_autooptimiser_level_horizon="-l"
                       else
                          cmd_autooptimiser_level_horizon="" 
                       end
                       if (check_button_auto_projection_size.value) then
                          cmd_autooptimiser_auto_projection="-s"
                       else
                          cmd_autooptimiser_auto_projection=""
                       end
                       cmd_autooptimiser_input="\""..tmppath.."/project.pto\""
                       cmd_autooptimiser_output="-o \""..tmppath.."/project.pto\""
                       autooptimiserStartCommand="autooptimiser "..cmd_autooptimiser_auto_align_mode.." "..cmd_autooptimiser_pairwise_optimisation.." "..cmd_autooptimiser_photometric_parameters.." "..cmd_autooptimiser_level_horizon.." "..cmd_autooptimiser_auto_projection.." "..cmd_autooptimiser_output.." "..cmd_autooptimiser_input
                       dt.print_error(autooptimiserStartCommand)
                       dt.print(_('optimizing image positions'))
                       result_autooptimiser=dt.control.execute(autooptimiserStartCommand)
                       dt.print_error("AUTOOPTIMIZER RETURN "..result_autooptimiser)
job.percent = 0.7              
                       if (result_autooptimiser == 0) then
                           -- create pano_modify command   
                           cmd_pano_modify_canvas="--canvas="..combobox_canvas_size.value
                           if (combobox_output_crop.selected == 3) then
                           cmd_pano_modify_crop=""
                           else 
                           cmd_pano_modify_crop="--crop="..combobox_output_crop.value
                           end
                           if (check_button_straighten.value) then
                               cmd_pano_modify_straighten="--straighten"
                           else
                               cmd_pano_modify_straighten=""
                           end
                           if (check_button_center.value) then
                               cmd_pano_modify_center="--center"
                           else
                               cmd_pano_modify_center=""
                           end
                           if (combobox_exposure.selected == 1) then
                               cmd_pano_modify_exposure="--output-exposure=auto"
                           else
                               cmd_pano_modify_exposure="--output-exposure="..(replace_comma_to_dot(slider_exposure.value))  
                           end
                           if (combobox_output_format.selected == 1) then
                               cmd_pano_modify_output_format="--ldr-file=TIF"
                               cmd_pano_modify_output_compression="--ldr-compression="..dt.preferences.read("modul_pano_pro", "compression_tiff", "enum")  
                           elseif (combobox_output_format.selected == 2) then
                               cmd_pano_modify_output_format="--ldr-file=JPG"
                               cmd_pano_modify_output_compression="--ldr-compression="..truncate(dt.preferences.read("pano_pro", "compression_jpeg", "integer"))  
                           elseif (combobox_output_format.selected == 3) then
                               cmd_pano_modify_output_format="--ldr-file=PNG"  
                           end
                           cmd_pano_modify_input="\""..tmppath.."/project.pto\""
                           cmd_pano_modify_output="-o \""..tmppath.."/project.pto\""
                           pano_modifyStartCommand="pano_modify --blender=ENBLEND "..cmd_pano_modify_canvas.." "..cmd_pano_modify_crop.." "..cmd_pano_modify_straighten.." "..cmd_pano_modify_center.." "..cmd_pano_modify_exposure.." "..cmd_pano_modify_output.." "..cmd_pano_modify_input
                           dt.print_error(pano_modifyStartCommand)
                           result_pano_modify=dt.control.execute(pano_modifyStartCommand)
 job.percent = 0.8                           
                           if (result_pano_modify == 0) then
                           -- create nona command     
                              cmd_nona_prj_input="\""..tmppath.."/project.pto\""
                              cmd_nona_bitmap_output="\""..tmppath.."/nona\""
                              
                              if (dt.preferences.read("pano_pro", "nona_use_gpu", "bool")) then    
                                  cmd_nona_use_gpu="-g"
                              else
                                  cmd_nona_use_gpu=""
                              end
                              if (combobox_exposure.selected == 1) then
                                 cmd_nona_exposure=""
                              else
                                 cmd_nona_exposure="-e "..(replace_comma_to_dot(slider_exposure.value))  
                              end
                              
                              nonaStartCommand="nona -v -m TIFF_m "..cmd_nona_use_gpu.." "..cmd_nona_exposure.." -o "..cmd_nona_bitmap_output.." "..cmd_nona_prj_input
                              dt.print_error(nonaStartCommand)
                              dt.print(_('merging images'))
                              result_nona=dt.control.execute(nonaStartCommand)   
                              dt.print_error("NONA RETURN "..result_nona)
job.percent = 0.9 
                              if (result_nona == 0) then                           
                               -- create enblend command   
                                 tiff_index=0
                                 for x=1, counted_images do
                                     if (tiff_index <=9) then
                                        nona_filename_index="000"..(tostring(tiff_index))
                                     elseif (tiff_index <=99) then
                                        nona_filename_index="00"..(tostring(tiff_index))    
                                     elseif (tiff_index <=999) then
                                        nona_filename_index="0"..(tostring(tiff_index))       
                                     end
                                     cmd_enblend_bitmap_input = cmd_enblend_bitmap_input.."\""..tmppath.."/nona"..nona_filename_index..".tif\" "
                                     tiff_index=tiff_index+1
                                  end
                                                                    
                                  if (combobox_output_format.selected == 1) then
                                    cmd_suffix_output_format="tif"
                                    cmd_enblend_output_compression="--compression "..dt.preferences.read("pano_pro", "compression_tiff", "enum")
                                    cmd_enblend_output_depth="-d "..dt.preferences.read("pano_pro", "image_color_depth", "enum")
                                  elseif (combobox_output_format.selected == 2) then
                                    cmd_suffix_output_format="jpg"
                                    cmd_enblend_output_compression="--compression "..dt.preferences.read("pano_pro", "compression_jpeg", "integer")
                                    cmd_enblend_output_depth=""
                                  elseif (combobox_output_format.selected == 3) then
                                    cmd_suffix_output_format="png"  
                                    cmd_enblend_output_compression=""
                                   cmd_enblend_output_depth=""
                                  end
    
                                 cmd_output_path=file_chooser_button_path_pano.value
                                 if(combobox_existing_file.selected == 1) then
                                      path_with_filename=cmd_output_path.."/"..(GetFileName(first_imagefile)).."-"..(GetFileName(last_imagefile)).."."..cmd_suffix_output_format
                                      while (file_exists(path_with_filename)) do
                                          cmd_output_image_index=cmd_output_image_index+1
                                          if cmd_output_image_index <= 9 then
                                              outputindex="0"..tostring(cmd_output_image_index)
                                          else
                                              outputindex=tostring(cmd_output_image_index)   
                                          end
                                          path_with_filename=cmd_output_path.."/"..(GetFileName(first_imagefile)).."-"..(GetFileName(last_imagefile)).."_"..outputindex.."."..cmd_suffix_output_format
                                      end 
                                 cmd_output_image_index=0
                                 cmd_enblend_bitmap_output="\""..path_with_filename.."\""
                                 else
                                    path_with_filename=cmd_output_path.."/"..(GetFileName(first_imagefile)).."-"..(GetFileName(last_imagefile)).."."..cmd_suffix_output_format
                                    cmd_enblend_bitmap_output="\""..path_with_filename.."\""
                                 end
                                 dt.print_error(cmd_enblend_bitmap_output)
                              
                                 if (dt.preferences.read("pano_pro", "nona_use_gpu", "bool")) then
                                    cmd_enblend_use_gpu="--gpu"
                                 else
                                   cmd_enblend_use_gpu=""   
                                 end 

                                 enblendStartCommand="enblend -v "..cmd_enblend_use_gpu.." "..cmd_enblend_output_compression.." "..cmd_enblend_output_depth.." -o "..cmd_enblend_bitmap_output.." "..cmd_enblend_bitmap_input
                                 dt.print_error(enblendStartCommand)
                                 result_enblend=dt.control.execute(enblendStartCommand)  
                                 dt.print_error("ENBLEND RETURN "..result_enblend)
                                  if ((result_enblend == 0) and (dt.preferences.read("pano_pro", "exiftool_copy_tags", "bool")) and (dt.preferences.read("pano_pro", "add_image_to_db", "bool"))) then        
                                  --attach exif data and add to db
                                        exifStartCommand="exiftool -TagsFromFile \""..first_imagefile.."\" -exif:all --subifd:all -overwrite_original "..cmd_enblend_bitmap_output..""
                                        dt.print_error(exifStartCommand)
                                        resultexif=dt.control.execute(exifStartCommand)
                                        local image = dt.database.import(path_with_filename)
                                        dt.print_error(path_with_filename)
                                        remove_tmp_files()
                                        job.valid = false
                                        dt.print(_('process successfully completed'))
                                  elseif ((result_enblend == 0) and (dt.preferences.read("pano_pro", "exiftool_copy_tags", "bool")) and (dt.preferences.read("pano_pro", "add_image_to_db", "bool") == false)) then        
                                  --attach exif data only   
                                        exifStartCommand="exiftool -TagsFromFile \""..first_imagefile.."\" -exif:all --subifd:all -overwrite_original "..cmd_enblend_bitmap_output..""
                                        dt.print_error(exifStartCommand)
                                        resultexif=dt.control.execute(exifStartCommand)
                                        dt.print_error("EXIFTOOL RETURN "..resultexif)
                                        if (resultexif == 0) then
                                            remove_tmp_files()
                                            job.valid = false 
                                            dt.print(_('process successfully completed'))
                                        else
                                            remove_tmp_files()
                                            job.valid = false
                                            dt.print(_('ERROR: exiftool doesn\'t work. for more informations see terminal output'))  
                                        end
                                  
                                  elseif ((result_enblend == 0) and (dt.preferences.read("pano_pro", "exiftool_copy_tags", "bool") == false) and (dt.preferences.read("pano_pro", "add_image_to_db", "bool"))) then   
                                  -- add to db only
                                     local image = dt.database.import(path_with_filename)
                                        dt.print_error(path_with_filename)
                                        remove_tmp_files()
                                        job.valid = false
                                        dt.print(_('process successfully completed'))
                                        
                                  elseif ((result_enblend == 0) and (dt.preferences.read("pano_pro", "exiftool_copy_tags", "bool") == false) and (dt.preferences.read("pano_pro", "add_image_to_db", "bool") == false)) then   
                                  -- exit without exif tags and add to db
                                        remove_tmp_files()
                                        job.valid = false
                                        dt.print(_('process successfully completed'))      
                                  elseif (result_enblend == 256) then
                                      remove_tmp_files()
                                      job.valid = false
                                    dt.print(_('ERROR: nona could not create any images. \nthe overlapping of the images may not be sufficient, \nor insufficient control points could be created. \nfor more informations see terminal output'))
                                  else
                                  -- enblend error
                                    dt.print(_('ERROR: enblend doesn\'t work. for more informations see terminal output'))
                                    remove_tmp_files()
                                    job.valid = false 
                                  end
                                else
                                    dt.print(_('ERROR: nona doesn\'t work. for more informations see terminal output'))
                                    remove_tmp_files()
                                    job.valid = false
                              end    
                              
                           else
                               dt.print(_('ERROR: pano_modify doesn\'t work. for more informations see terminal output'))  
                               remove_tmp_files()
                               job.valid = false
                           end
                       else
                          dt.print(_('ERROR: autooptimiser doesn\'t work. for more informations see terminal output'))  
                          remove_tmp_files()
                          job.valid = false
                       end
                    else
                      dt.print(_('ERROR: linefind doesn\'t work. for more informations see terminal output'))
                      remove_tmp_files()
                      job.valid = false
                    end
               else
                  dt.print(_('ERROR: cpclean doesn\'t work. for more informations see terminal output')) 
                  remove_tmp_files()
                  job.valid = false
               end
           else
              dt.print(_('ERROR: cpfind doesn\'t work. for more informations see terminal output')) 
              remove_tmp_files()
              job.valid = false
           end
      else
         dt.print(_('ERROR: pto_gen doesn\'t work. for more informations see terminal output'))
         remove_tmp_files()
         job.valid = false
      end
      
    end 
end


-- limit export to tiff
local function support_format(storage, format)
  fmt = string.lower(format.name)
  if string.match(fmt,"tiff") == nil then
    return false
  else
    return true
  end   
end  



-- REGISTER
dt.register_storage("pano_pro", _('panorama image'), show_status, create_pano_image, support_format, nil, widget)


                        
dt.preferences.register("pano_pro",        
                        "keypoint_cache",                                          -- name
                        "bool",                                                   -- type
                        _('pano pro: keypoint cache'),                       -- label
                        _('use cache by keypoint calculation'),                  -- tooltip
                        true)                                                    -- default
                        
dt.preferences.register("pano_pro",        
                        "nona_use_gpu",                                          -- name
                        "bool",                                                   -- type
                        _('pano pro: use GPU for remaping'),                       -- label
                        _('set the GPU remapping for image align'),                  -- tooltip
                        false)                                                    -- default                        
                        
dt.preferences.register("pano_pro",        
                        "exiftool_copy_tags",                                     -- name
                        "bool",                                                   -- type
                        _('pano pro: copy exif tags'),                             -- label
                        _('copy the exif tags from the first image to the target'),  -- tooltip
                        true)                                                     -- default                        

dt.preferences.register("pano_pro",        
                        "add_image_to_db",                                        -- name
                        "bool",                                                   -- type
                        _('pano pro: add panorama image to database'),                -- label
                        _('add the panorama image to the darktable database'),          -- tooltip
                        false)                                                    -- default           
                        
                        
dt.preferences.register("pano_pro",        
                        "compression_jpeg",                        -- name
                        "integer",                                 -- type
                        _('pano pro: JPEG compression'),            -- label
                        _('set the compression for JPEG files'),      -- tooltip
                        98,                                        -- default
                        50,                                        -- min
                        100)                                       -- max
                        
dt.preferences.register("pano_pro",        
                        "compression_tiff",                        -- name
                        "enum",                                    -- type
                        _('pano pro: TIFF compression'),            -- label
                        _('set the compression type for tiff files'), -- tooltip
                        "LZW",                                     -- default
                        "NONE", "DEFLATE","LZW","PACKBITS")        -- values

                    
                        
dt.preferences.register("pano_pro",        
                        "image_color_depth",                       -- name
                        "enum",                                    -- type
                        _('pano pro: image color depth (bit)'),     -- label
                        _('set the output color depth'),              -- tooltip
                        "16",                                      -- default
                        "8","16")                                  -- values