--[[
    gpsbabel plugin for darktable 2.2.X
    
    darktable is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    darktable is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with darktable.  If not, see <http://www.gnu.org/licenses/>.
]]
--[[
Version 2.2.2
* it creates a new GPS Logfile downloader module at the lighttable
* the module can download and convert logfiles
* the module is a frontend for gpsbabel (No complete support)
* supported devices:
   - Wintec WBT-100/200 GPS
   - Holux M-241 (MTK based)
   - Garmin serial/USB protocol 
   - GlobalSat DG-100/BT-335
   - GlobalSat DG-200
   - Brauniger IQ Series Barograph
   - MTK Logger (iBlue 747,Qstarz BT-1000,...
   - NaviGPS GT-11/BGT-11
   - SkyTraq Venus based loggers
* supported file fomats
   - GPX
   - CVS
   - KML
   - And many more. For more information see http://www.gpsbabel.org
   
DEPENDS
- gpsbabel version 1.5.4 or newer

Release notes 
* fix compatibility with other scripts
                
USAGE
* 
- Select your device or file
- Select your interface or input file with file format 
- Select the download option, file format, directory and create a filename without suffix
- click download and wait for the download confirmation
                
                
]]

local dt = require "darktable"
local gettext = dt.gettext

-- works with darktable 2.2.X
dt.configuration.check_version(...,{4,0,0})

-- command variables
gpsbabel_cmd_device=""
gpsbabel_cmd_interface=""            --  Interface or File -f /dev/cu.WBT200-SPPslave-1 or -f /tmp/test.gpx
gpsbabel_cmd_inputformat=""          --  Input device of fileformat -i gpx or -i wbt
gpsbabel_cmd_outputformat=""         --  Outputfileformat  -o gpx
gpsbabel_cmd_outputfile=""
gpsbabel_cmd_outputsuffix=""
gpsbabel_cmd_outputpath=""            
gpsbabel_cmd_waypoints=""             --  Process waypoint information [default] -w
gpsbabel_cmd_tracks=""                --  Process track information -t
gpsbabel_cmd_routes=""                --  Process route information -r
gpsbabel_cmd_erase_after_download=""
gpsbabel_cmd_interpolate=""           --  filtername -x 
gpsbabelStartCommand=""

-- initialize combobox and checkbox value for default (Wintec WBT-100/200 GPS)
if dt.preferences.read("gpsbabel",  "selected_device", "integer") == 0 then

   dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)       --Interface is selectable
   dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     --Inputfile is not selectable 
   dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   --Inputfileformat is not selectable             
   dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)       --Checkbox not selectable
   dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)          --Checkbox not selectable
   dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)         --Checkbox not selectable
   dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", true)     --Checkbox not selectable
   dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false)     --Button not selectable
   dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true)     --Interpolate yes
    
   dt.preferences.write("gpsbabel", "selected_device", "integer", 1)           --Default device Wintec WBT-100/200 GPS
   dt.preferences.write("gpsbabel", "selected_interface", "integer", 2)        --Default interface /dev/ttyUSB0
   dt.preferences.write("gpsbabel", "selected_input_format", "integer", 1)     --Default input file format gpx            
   dt.preferences.write("gpsbabel", "selected_waypoints", "bool", true)        --Wintec WBT-100/200 GPS waypoints are supported
   dt.preferences.write("gpsbabel", "selected_tracks", "bool", true)           --Wintec WBT-100/200 GPS tracks are supported
   dt.preferences.write("gpsbabel", "selected_routes", "bool", false)          --Wintec WBT-100/200 GPS routes are nit spported   
   dt.preferences.write("gpsbabel", "selected_erase_after", "bool", true)      --Wintec WBT-100/200 GPS supports erase data after download
   dt.preferences.write("gpsbabel", "selected_interpolate", "bool", true)      --Interpolate wapoints   
   dt.preferences.write("gpsbabel", "selected_output_format", "integer", 1)    --GPX format

end

-- Tell gettext where to find the .mo file translating messages for a particular domain
gettext.bindtextdomain("gpsbabel",dt.configuration.config_dir.."/lua/")

local function _(msgid)
    return gettext.dgettext("gpsbabel", msgid)
end


-- check installation
local function checkIfBinExists(bin)
  local handle = io.popen("which "..bin)
  local result = handle:read()
  local ret
  handle:close()
  if (result) then
 --   dt.print_error("true checkIfBinExists: "..bin)
    ret = true
  else
    dt.print_error(bin.." not found")
    ret = false
  end


  return ret
end



-- start GUI
gpsbabel_label_source = dt.new_widget("label")
{
     label = _('source:'),
     ellipsize = "start",
     halign = "start"
}

gpsbabel_label_filename = dt.new_widget("label")
{
     label = _('filename without suffix:'),
     ellipsize = "start",
     halign = "start"
}
                
               
gpsbabel_label_download = dt.new_widget("label")
{
     label = _('options:'),
     ellipsize = "start",
     halign = "start"
}                
    
gpsbabel_label_path = dt.new_widget("label")
{
     label = _('target:'),
     ellipsize = "start",
     halign = "start"
}
                
gpsbabel_label_filter = dt.new_widget("label")
{
     label = _('data filter:'),
     ellipsize = "start",
     halign = "start"
}
                
                
gpsbabel_separator1 = dt.new_widget("separator")
{}

gpsbabel_separator2 = dt.new_widget("separator")
{}                
                
gpsbabel_separator3 = dt.new_widget("separator")
{}                  
                
    




gpsbabel_check_button_waypoints = dt.new_widget("check_button")
{
    label = _('waypoints'), 
    value = dt.preferences.read("gpsbabel", "selected_waypoints", "bool"),
    sensitive = dt.preferences.read("gpsbabel", "sensitive_waypoints", "bool"),
    tooltip = _('download waypoints'),  
    clicked_callback = function(way)   
      --check device feature
      if (dt.preferences.read("gpsbabel", "sensitive_waypoints", "bool")) then
          if (way.value) then
               dt.preferences.write("gpsbabel", "selected_waypoints", "bool", true)
               gpsbabel_cmd_waypoints="-w"
          --   return
          else
              dt.preferences.write("gpsbabel", "selected_waypoints", "bool", false)
              gpsbabel_cmd_waypoints=""
         --    return
          end
      else
         gpsbabel_check_button_waypoints.sensitive = false
         gpsbabel_check_button_waypoints.value = false
         gpsbabel_cmd_waypoints=""
      end
    end,
    
    reset_callback = function(self) 
       self.sensitive = true
       self.value = true
       dt.preferences.write("gpsbabel", "selected_waypoints", "bool", true)
       gpsbabel_cmd_waypoints="-w"
    end
}
    

gpsbabel_check_button_tracks = dt.new_widget("check_button")
{
    label = _('tracks'), 
    value = dt.preferences.read("gpsbabel", "selected_tracks", "bool"),
    sensitive = dt.preferences.read("gpsbabel", "sensitive_tracks", "bool"),
    tooltip = _('download tracks'),  
    clicked_callback = function(tra)  
       --check device feature
       if (dt.preferences.read("gpsbabel", "sensitive_tracks", "bool")) then
          if (tra.value) then
               dt.preferences.write("gpsbabel", "selected_tracks", "bool", true)
               gpsbabel_cmd_tracks="-t"
             return
          else
              dt.preferences.write("gpsbabel", "selected_tracks", "bool", false)
              gpsbabel_cmd_tracks=""
             return
          end
       else
         gpsbabel_check_button_tracks.sensitive = false
         gpsbabel_check_button_tracks.value = false
         gpsbabel_cmd_tracks=""  
       end
    end,
    reset_callback = function(self) 
       self.sensitive = true
       self.value = true
       dt.preferences.write("gpsbabel", "selected_tracks", "bool", true)
       gpsbabel_cmd_tracks="-t"
    end

}


gpsbabel_check_button_routes = dt.new_widget("check_button")
{
    label = _('routes'), 
    value = dt.preferences.read("gpsbabel", "selected_routes", "bool"),
    sensitive = dt.preferences.read("gpsbabel", "sensitive_routes", "bool"),
    tooltip = _('download routes'),
    clicked_callback = function(rou)   
       --check device feature
       if (dt.preferences.read("gpsbabel", "sensitive_routes", "bool")) then
          if (rou.value) then
               dt.preferences.write("gpsbabel", "selected_routes", "bool", true)
               gpsbabel_cmd_routes="-r"
             return
          else
              dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
              gpsbabel_cmd_routes=""
             return
          end
       else
         gpsbabel_check_button_routes.sensitive = false
         gpsbabel_check_button_routes.value = false
         gpsbabel_cmd_routes=""    
       end    
    end,
    reset_callback = function(self) 
       self.sensitive = false
       self.value = false
       dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
       gpsbabel_cmd_routes=""
       dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
    end
}

gpsbabel_check_button_delete_after = dt.new_widget("check_button")
{
    label = _('delete GPS logfiles after download'), 
    value = false,
    tooltip = _('delete the logfiles from the source after download'),
    sensitive = dt.preferences.read("gpsbabel", "sensitive_erase_after", "bool"),            
    reset_callback = function(self_delete_after) 
       self_delete_after.value = false
       self_delete_after.sensitive = dt.preferences.read("gpsbabel", "sensitive_erase_after", "bool")
    end,
    clicked_callback = function(execute_delete)
       --check device feature
       if (dt.preferences.read("gpsbabel", "sensitive_erase_after", "bool")) then
           if (execute_delete.value) then
             gpsbabel_cmd_erase_after_download=",erase"
           else
             gpsbabel_cmd_erase_after_download=""  
         end
       else
         gpsbabel_check_button_delete_after.sensitive = false
         gpsbabel_check_button_delete_after.value = false
         gpsbabel_cmd_routes=""    
       end  
    end
}
    

gpsbabel_check_button_interpolate = dt.new_widget("check_button")
{
    label = _('interpolate trackpoints'), 
    value = dt.preferences.read("gpsbabel", "selected_interpolate", "bool"),
    tooltip = _('this option adds every second a trackpoint between two logged trackpoints'),
    reset_callback = function(interself) 
       interself.value = true
       interself.sensitive = true
    end,
    clicked_callback = function(inter)
       --check device feature               
       if (dt.preferences.read("gpsbabel", "sensitive_interpolate", "bool")) then
           if (inter.value) then
             dt.preferences.write("gpsbabel", "selected_interpolate", "bool", true)
             gpsbabel_cmd_interpolate="interpolate,time=1"
           else
             dt.preferences.write("gpsbabel", "selected_interpolate", "bool", false)
             gpsbabel_cmd_interpolate=""  
         end
       else
         inter.sensitive = false
         inter.value = false
         gpsbabel_cmd_interpolate=""    
       end  
    end
}                
                            

gpsbabel_file_chooser_input_button = dt.new_widget("file_chooser_button")
{
    title = _('source file'),  -- The title of the window when choosing a file
    value = "",                       -- The currently selected file
    tooltip = _('please select your source file'),
    is_directory = false,              -- True if the file chooser button only allows directories to be selecte
    sensitive = dt.preferences.read("gpsbabel", "sensitive_input_file", "bool"),
    changed_callback = function(input_path)             
      gpsbabel_cmd_interface = "-f "..input_path.value 
   --   gpsbabel_combobox_input.value = dt.preferences.read("gpsbabel", "selected_input_format", "integer")   PRÜFEN OB NOTWENDIG
    end
                
}                
                

                
gpsbabel_erase_only_button = dt.new_widget("button")
    {
      label = _('delete source data without download'),
      sensitive = dt.preferences.read("gpsbabel", "sensitive_erase_only", "bool"),
      clicked_callback = function (_)
         -- check device feature
         if (dt.preferences.read("gpsbabel", "sensitive_erase_only", "bool") and gpsbabel_combobox_interface.sensitive) then
              -- check gpsbabel installation
              if (checkIfBinExists("gpsbabel")) then
               -- create command
                   gpsbabel_cmd_interface = "-f " ..gpsbabel_combobox_interface.value
                   gpsbabelStartCommand="gpsbabel " ..gpsbabel_cmd_device..",erase_only " ..gpsbabel_cmd_interface
                   dt.print_error(gpsbabelStartCommand)    
                   local result = nil
                   dt.print(_('erase log data on the device...'))
                   result = dt.control.execute( gpsbabelStartCommand)
                    if (result == 0) then
                     dt.print(_('erase log data completed successfully'))
                   else
                     dt.print(_('ERROR: please check the device, interface and connection to the device. For more information see terminal output')) 
                   end
              else    
                  dt.print(_('ERROR: gpsbabel not found. Please install gpsbabel.'))
                  dt.print_error(_('gpsbabel not found. Please install gpsbabel.'))
                --  return
              end
         else
           gpsbabel_erase_only_button.sensitive = false
         end
      end
    }                
                
                
                



gpsbabel_file_chooser_output_button = dt.new_widget("file_chooser_button")
{
    title = _('download GPS data'),  -- The title of the window when choosing a file
    is_directory = true,  -- True if the file chooser button only allows directories to be selecte
    tooltip = _('select the directory for the target file'),
    changed_callback = function(path)
       dt.preferences.write("gpsbabel", "path", "string", gpsbabel_file_chooser_output_button.value)
       gpsbabel_cmd_outputpath= gpsbabel_file_chooser_output_button.value.."/"
    end

}



gpsbabel_entry_filename = dt.new_widget("entry")
{
    text = "$(YEAR)-$(MONTH)-$(DAY)", 
    sensitive = true,
    is_password = true,
    editable = true,
    tooltip = _('enter the filename without suffix. the filename $(YEAR)-$(MONTH)-$(DAY) creates a file like 2017-02-22.gpx'),
    reset_callback = function(self) 
       self.text = "$(YEAR)-$(MONTH)-$(DAY)" 
    end

}

gpsbabel_combobox_timezone = dt.new_widget("combobox")
{
    label = _('rectify time zone'), 
    tooltip = _('this option can rectify the time zone from your GPS device'),
    editable = false,
    value = 24, "-23h","-22h","-21h","-20h","-19h","-18h","-17h","-16h","-15h","-14h","-13h","-12h","-11h","-10h","-9h","-8h","-7h","-6h","-5h","-4h","-3h","-2h","-1h","0h","+1h","+2h","+3h","+4h","+5h","+6h","+7h","+8h","+9h","+10h","+11h","+12h","+13h","+14h","+15h","+16h","+17h","+18h","+19h","+20h","+21h","+22h","+23h",
    reset_callback = function(reset_timezone) 
       reset_timezone.sensitive = true
       reset_timezone.value = 24
    end, 
                
}                
                
                
gpsbabel_combobox_output = dt.new_widget("combobox")
{
    label = _('file format'), 
    tooltip = _('please select the output format'),
    editable = false,
    value = dt.preferences.read("gpsbabel", "selected_output_format", "integer"),
    changed_callback = function(output_path) 
       gpsbabel_cmd_outputformat = "-o "..output_path.value
       gpsbabel_cmd_outputsuffix = "."..output_path.value
       dt.preferences.write("gpsbabel", "selected_output_format", "integer", output_path.selected)
    end,
    "gpx", "kml","csv","html",
    
    reset_callback = function(reset) 
       reset.value = 1
       gpsbabel_cmd_outputformat = "-o gpx" 
    end,
}
                
gpsbabel_combobox_interface = dt.new_widget("combobox")
{
    label = _('interface'), 
    tooltip = _('please selecting your interface'),
    editable = true,
    sensitive = dt.preferences.read("gpsbabel", "sensitive_interface", "bool"),           
    value = dt.preferences.read("gpsbabel", "selected_interface", "integer"),
    changed_callback = function(selinter) 
      gpsbabel_cmd_interface = "-f "..selinter.value
      dt.preferences.write("gpsbabel", "selected_interface", "integer", selinter.selected)
    end,
    "usb:", "/dev/ttyUSB0", "com1:","/dev/cu.WBT200-SPPslave-1","/dev/rfcomm0","/dev/ttyS0",
    
    reset_callback = function(interfacereset) 
       interfacereset.value = 2
       gpsbabel_cmd_interface = "-f " ..gpsbabel_combobox_interface.value
    end,   
}


function setsensitive()
 
    gpsbabel_combobox_interface.sensitive = dt.preferences.read("gpsbabel", "sensitive_interface", "bool")    
    gpsbabel_file_chooser_input_button.sensitive = dt.preferences.read("gpsbabel", "sensitive_input_file", "bool")
    gpsbabel_combobox_input.sensitive = dt.preferences.read("gpsbabel", "sensitive_input_format", "bool")
    gpsbabel_erase_only_button.sensitive = dt.preferences.read("gpsbabel", "sensitive_erase_only", "bool") 
    gpsbabel_check_button_waypoints.sensitive = dt.preferences.read("gpsbabel", "sensitive_waypoints", "bool")
    gpsbabel_check_button_tracks.sensitive =  dt.preferences.read("gpsbabel", "sensitive_tracks", "bool")
    gpsbabel_check_button_routes.sensitive = dt.preferences.read("gpsbabel", "sensitive_routes", "bool")
    gpsbabel_check_button_delete_after.sensitive = dt.preferences.read("gpsbabel", "sensitive_erase_after", "bool") 
                
    if  (gpsbabel_check_button_waypoints.sensitive) then           
      gpsbabel_check_button_waypoints.value = true
    else
      gpsbabel_check_button_waypoints.value = false
    end
    
    if  (gpsbabel_check_button_tracks.sensitive) then           
      gpsbabel_check_button_tracks.value = true
    else
      gpsbabel_check_button_tracks.value = false
    end            
                
    if  (gpsbabel_check_button_routes.sensitive) then           
      gpsbabel_check_button_routes.value = true
    else
      gpsbabel_check_button_routes.value = false
    end              
                
    if  (gpsbabel_check_button_delete_after.sensitive) then           
      gpsbabel_check_button_delete_after.value = true
    else
      gpsbabel_check_button_delete_after.value = false
    end              
                
end                               
                
                
 gpsbabel_combobox_input = dt.new_widget("combobox")
{
    label = _('file format'), 
    tooltip = _('please select the input format. more informations see gpsbabel homepage'),
    editable = false,
    sensitive = dt.preferences.read("gpsbabel", "sensitive_input_format", "bool"),
    value = dt.preferences.read("gpsbabel", "selected_input_format", "integer"),
    changed_callback = function(input) 
        if (input.value == "GPX exchange format (gpx)") then
          gpsbabel_cmd_device = "-i gpx"
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()     
                
        elseif (input.value == "Keyhole Markup Language (kml)") then   
          gpsbabel_cmd_device = "-i kml"
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()  
        elseif (input.value == "Comma-separated values (csv)") then                  
          gpsbabel_cmd_device = "-i csv" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()  
         elseif (input.value == "Universal csv with field structure (csv)") then                  
          gpsbabel_cmd_device = "-i unicsv" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()  
        elseif (input.value == "Alan Map500 tracklogs") then                  
          gpsbabel_cmd_device = "-i alantrl" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
        elseif (input.value == "Alan Map500 waypoints and routes") then                  
          gpsbabel_cmd_device = "-i alanwpr" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "Bushnell GPS Trail file") then                  
          gpsbabel_cmd_device = "-i bushnell_trl" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
         elseif (input.value == "Bushnell GPS Waypoint file") then                  
          gpsbabel_cmd_device = "-i bushnell" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
        elseif (input.value == "Cambridge/Winpilot glider software") then                  
          gpsbabel_cmd_device = "-i cambridge" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
        elseif (input.value == "CarteSurTable data file") then                  
          gpsbabel_cmd_device = "-i cst" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
         elseif (input.value == "Columbus/Visiontac V900 files (.csv)") then                  
          gpsbabel_cmd_device = "-i v900" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
         elseif (input.value == "CompeGPS data files (.wpt/.trk/.rte)") then                  
          gpsbabel_cmd_device = "-i compegps" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
         elseif (input.value == "Data Logger iBlue747 csv") then                  
          gpsbabel_cmd_device = "-i iblue747" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
        elseif (input.value == "Data Logger iBlue757 csv") then                  
          gpsbabel_cmd_device = "-i iblue757" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()       
        elseif (input.value == "DeLorme .an1 (drawing) file") then                  
          gpsbabel_cmd_device = "-i an1" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
        elseif (input.value == "DeLorme GPL") then                  
          gpsbabel_cmd_device = "-i gpl" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()     
        elseif (input.value == "DeLorme Street Atlas Plus") then                  
          gpsbabel_cmd_device = "-i saplus" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "DeLorme Street Atlas Route") then                  
          gpsbabel_cmd_device = "-i saroute" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()   
        elseif (input.value == "DeLorme XMap HH Native .WPT") then                  
          gpsbabel_cmd_device = "-i xmap" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
        elseif (input.value == "DeLorme XMap/SAHH 2006 Native .TXT") then                  
          gpsbabel_cmd_device = "-i xmap2006" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "DeLorme XMat HH Street Atlas USA .WPT (PPC)") then                  
          gpsbabel_cmd_device = "-i xmapwpt" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()     
        elseif (input.value == "Destinator Itineraries (.dat)") then                  
          gpsbabel_cmd_device = "-i destinator_itn" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
         elseif (input.value == "Destinator Points of Interest (.dat)") then                  
          gpsbabel_cmd_device = "-i destinator_poi" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()  
        elseif (input.value == "Destinator TrackLogs (.dat)") then                  
          gpsbabel_cmd_device = "-i destinator_trl" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()  
        elseif (input.value == "EasyGPS binary format") then                  
          gpsbabel_cmd_device = "-i easygps" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
        elseif (input.value == "Embedded Exif-GPS data (.jpg)") then                  
          gpsbabel_cmd_device = "-i exif" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "Energympro GPS training watch") then                  
          gpsbabel_cmd_device = "-i energympro" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
        elseif (input.value == "Flexible and Interoperable Data Transfer (FIT) Activity file") then                  
          gpsbabel_cmd_device = "-i garmin_fit" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive() 
        elseif (input.value == "Fugawi") then                  
          gpsbabel_cmd_device = "-i fugawi" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
        elseif (input.value == "G7ToWin data files (.g7t)") then                  
          gpsbabel_cmd_device = "-i g7towin" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "Garmin 301 Custom position and heartrate") then                  
          gpsbabel_cmd_device = "-i garmin301" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
       elseif (input.value == "Garmin G1000 datalog input filter file") then                  
          gpsbabel_cmd_device = "-i garmin_g1000" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()            
        elseif (input.value == "Garmin Logbook XML") then                  
          gpsbabel_cmd_device = "-i glogbook" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()           
         elseif (input.value == "Garmin MapSource - gdb") then                  
          gpsbabel_cmd_device = "-i gdb" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
         elseif (input.value == "Garmin MapSource - mps") then                  
          gpsbabel_cmd_device = "-i mapsource" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()  
        elseif (input.value == "Garmin MapSource - txt (tab delimited)") then                  
          gpsbabel_cmd_device = "-i garmin_txt" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()      
         elseif (input.value == "Garmin PCX5") then                  
          gpsbabel_cmd_device = "-i pcx" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "Garmin POI database") then                  
          gpsbabel_cmd_device = "-i garmin_poi" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
         elseif (input.value == "Garmin Points of Interest (.gpi)") then                  
          gpsbabel_cmd_device = "-i garmin_gpi" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
        elseif (input.value == "Garmin Training Center (.tcx,.tcx)") then                  
          gpsbabel_cmd_device = "-i gtrnctr" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()
       elseif (input.value == "Geocaching.com .loc") then                  
          gpsbabel_cmd_device = "-i geo" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()            
        elseif (input.value == "GlobalSat GH625XT GPS training watch") then                  
          gpsbabel_cmd_device = "-i globalsat" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()       
        elseif (input.value == "Google Directions XML") then                  
          gpsbabel_cmd_device = "-i googledir" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
         elseif (input.value == "Google Navigator Tracklines (.trl)") then                  
          gpsbabel_cmd_device = "-i gnav_trl" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
         elseif (input.value == "GoPal GPS track log (.trk)") then                  
          gpsbabel_cmd_device = "-i gopal" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
         elseif (input.value == "GPS Tracking Key Pro text") then                  
          gpsbabel_cmd_device = "-i land_air_sea" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()   
         elseif (input.value == "GPS TrackMaker") then                  
          gpsbabel_cmd_device = "-i gtm" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
         elseif (input.value == "GPSman") then                  
          gpsbabel_cmd_device = "-i gpsman" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive() 
         elseif (input.value == "gpsutil") then                  
          gpsbabel_cmd_device = "-i gpsutil" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
        elseif (input.value == "HikeTech") then                  
          gpsbabel_cmd_device = "-i hiketech" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
        elseif (input.value == "Holux (gm-100) .wpo Format") then                  
          gpsbabel_cmd_device = "-i holux" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive() 
        elseif (input.value == "Holux M-241 (MTK based) Binary File Format") then                  
          gpsbabel_cmd_device = "-i m241-bin" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "Jelbert GeoTagger data file") then                  
          gpsbabel_cmd_device = "-i jtr" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
         elseif (input.value == "Lowrance USR") then                  
          gpsbabel_cmd_device = "-i lowranceusr" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
         elseif (input.value == "Lowrance USR version 4") then                  
          gpsbabel_cmd_device = "-i lowranceusr4" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()  
        elseif (input.value == "Magellan Explorist Geocaching") then                  
          gpsbabel_cmd_device = "-i maggeo" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
        elseif (input.value == "Magellan SD files (as for eXplorist)") then                  
          gpsbabel_cmd_device = "-i magellanx" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
         elseif (input.value == "Magellan SD files (as for Meridian)") then                  
          gpsbabel_cmd_device = "-i magellan" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "Magellan serial protocol") then                  
          gpsbabel_cmd_device = "-i magellan" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()       
        elseif (input.value == "Mainnav") then                  
          gpsbabel_cmd_device = "-i mainnav" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "Navigon Mobile Navigator .rte files") then                  
          gpsbabel_cmd_device = "-i nmn4" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
         elseif (input.value == "Navigon Waypoints") then                  
          gpsbabel_cmd_device = "-i navigonwpt" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
         elseif (input.value == "OpenStreetMap data files") then                  
          gpsbabel_cmd_device = "-i osm" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
         elseif (input.value == "Wintec TES file") then                  
          gpsbabel_cmd_device = "-i wintec_tes" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()   
         elseif (input.value == "Wintec WBT-100/200 Binary File Format") then                  
          gpsbabel_cmd_device = "-i wbt-bin" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
         elseif (input.value == "Wintec WBT-201/G-Rays 2 Binary File Format (tk1)") then                  
          gpsbabel_cmd_device = "-i wbt-tk1" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
         elseif (input.value == "XAiOX iTrackU Logger") then                  
          gpsbabel_cmd_device = "-i itracku" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()    
        elseif (input.value == "Yahoo Geocode API data") then                  
          gpsbabel_cmd_device = "-i yahoo" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()       
        end        

       dt.preferences.write("gpsbabel", "selected_input_format", "integer", input.selected)
    end,
    "GPX exchange format (gpx)", "Keyhole Markup Language (kml)","Comma-separated values (csv)","Universal csv with field structure (csv)","Alan Map500 tracklogs","Alan Map500 waypoints and routes","Bushnell GPS Trail file","Bushnell GPS Waypoint file","Cambridge/Winpilot glider software","CarteSurTable data file","Columbus/Visiontac V900 files (.csv)","CompeGPS data files (.wpt/.trk/.rte)","Data Logger iBlue747 csv","Data Logger iBlue757 csv","DeLorme .an1 (drawing) file","DeLorme GPL","DeLorme Street Atlas Plus","DeLorme Street Atlas Route","DeLorme XMap HH Native .WPT","DeLorme XMap/SAHH 2006 Native .TXT","DeLorme XMat HH Street Atlas USA .WPT (PPC)","Destinator Itineraries (.dat)","Destinator Points of Interest (.dat)","Destinator TrackLogs (.dat)","EasyGPS binary format","Embedded Exif-GPS data (.jpg)","Energympro GPS training watch","Flexible and Interoperable Data Transfer (FIT) Activity file","Fugawi","G7ToWin data files (.g7t)","Garmin 301 Custom position and heartrate","Garmin G1000 datalog input filter file","Garmin Logbook XML","Garmin MapSource - gdb","Garmin MapSource - mps","Garmin MapSource - txt (tab delimited)","Garmin PCX5","Garmin POI database","Garmin Points of Interest (.gpi)","Garmin Training Center (.tcx)","Garmin Training Center (.xml,.tcx)","Geocaching.com .loc","GlobalSat GH625XT GPS training watch","Google Directions XML","Google Navigator Tracklines (.trl)","GoPal GPS track log (.trk)","GPS Tracking Key Pro text","GPS TrackMaker","GPSman","gpsutil","HikeTech","Holux (gm-100) .wpo Format","Holux M-241 (MTK based) Binary File Format","Jelbert GeoTagger data file","Lowrance USR","Lowrance USR version 4","Magellan Explorist Geocaching","Magellan SD files (as for eXplorist)","Magellan SD files (as for Meridian)","Magellan serial protocol","Mainnav","Navigon Mobile Navigator .rte files","Navigon Waypoints","OpenStreetMap data files","Wintec TES file","Wintec WBT-100/200 Binary File Format","Wintec WBT-201/G-Rays 2 Binary File Format (tk1)","XAiOX iTrackU Logger","Yahoo Geocode API data",
                
    reset_callback = function(reset_input) 
       reset_input.value = 1
       cmd_inputdevice = "-i gpx" 
    end,
}               
                
gpsbabel_combobox_device = dt.new_widget("combobox")
{
    label = _('device or file'), 
    tooltip = _('please select your device or choose file'),
    value = dt.preferences.read("gpsbabel", "selected_device", "integer"),
    changed_callback = function(sd) 
       dt.preferences.write("gpsbabel", "selected_device", "integer", sd.selected) --sd.selected ist eine integer Zahl
       if (sd.value == "Wintec WBT-100/200 GPS") then
          gpsbabel_cmd_device="-i wbt"
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
                    
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false)   
          setsensitive()  
          
       elseif  (sd.value == "Holux M-241 (MTK based)") then  
          gpsbabel_cmd_device="-i m241"
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          
          dt.preferences.write("gpsbabel", "selected_waypointes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false) 
          setsensitive() 
    
        elseif  (sd.value == "Garmin serial/USB protocol") then  
          gpsbabel_cmd_device="-i garmin"
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false) 
          setsensitive() 
          
        elseif  (sd.value == "GlobalSat DG-100/BT-335") then  
          gpsbabel_cmd_device="-i dg-100"
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false) 
          setsensitive()  
          
        elseif  (sd.value == "GlobalSat DG-200") then  
          gpsbabel_cmd_device="-i dg-200" 
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false) 
          setsensitive()  
          
        elseif  (sd.value == "Brauniger IQ Series Barograph") then 
          gpsbabel_cmd_device="-i baroiq"
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
                    
          dt.preferences.write("gpsbabel", "selected_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false)
          setsensitive()
          
        elseif  (sd.value == "MTK Logger (iBlue 747,Qstarz BT-1000,...)") then
          gpsbabel_cmd_device="-i mtk"   
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false) 
          setsensitive()   
          
           
        elseif  (sd.value == "NaviGPS GT-11/BGT-11") then   
          gpsbabel_cmd_device="-i navilink"
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true)  
          
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false) 
          setsensitive()  
          
        elseif  (sd.value == "SkyTraq Venus based loggers") then
          gpsbabel_cmd_device="-i skytraq"
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false)
          setsensitive()  
        
        elseif  (sd.value == "File") then
          gpsbabel_cmd_device="-i "..dt.preferences.read("gpsbabel", "selected_input_format", "integer")
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", true)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", true)                       
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false)       
           
           --set sensitive (sen_interface,sen_inputfile,sen_inputformat,sen_eraseonly,sen_waypoints,sen_tracks,sen_routes,sen_deleteafter)         
          setsensitive()     
       end

    end,
    "Wintec WBT-100/200 GPS", "Holux M-241 (MTK based)", "Garmin serial/USB protocol", "GlobalSat DG-100/BT-335", "GlobalSat DG-200", "Brauniger IQ Series Barograph", "MTK Logger (iBlue 747,Qstarz BT-1000,...)", "NaviGPS GT-11/BGT-11", "SkyTraq Venus based loggers", "File",
   reset_callback = function(device_reset) 
       device_reset.value = 1  --Default device Wintec WBT-100/200 GPS
    end,
 

}
                         
                
gpsbabel_download_button = dt.new_widget("button")
    {
      label = _('process GPS logfile'),
      clicked_callback = function (_)
            -- check gpsbabel installation
           if (checkIfBinExists("gpsbabel")) then
               -- check download options
               if ((gpsbabel_check_button_waypoints.value) or (gpsbabel_check_button_tracks.value) or (gpsbabel_check_button_routes.value)) then
                    -- check path
                    if (gpsbabel_file_chooser_output_button.value ~= nil) then
                        gpsbabel_cmd_outputpath = gpsbabel_file_chooser_output_button.value
                         -- check filename
                           if (not (gpsbabel_entry_filename.text == "")) then
                              if (gpsbabel_entry_filename.text == "$(YEAR)-$(MONTH)-$(DAY)") then
                                 local YYYY = tostring(os.date("%Y"))
                                 local MM = tostring(os.date("%m"))
                                 local DD = tostring(os.date("%d"))
                                 gpsbabel_cmd_outputfile = YYYY.. "-" ..MM.. "-" ..DD..gpsbabel_cmd_outputsuffix
                               else
                                gpsbabel_cmd_outputfile = gpsbabel_entry_filename.text..gpsbabel_cmd_outputsuffix
                               end
                               
                               -- check options and filters
                               if (gpsbabel_check_button_waypoints.value) then
                                 dt.preferences.write("gpsbabel", "selected_waypoints", "bool", true)
                                 gpsbabel_cmd_waypoints="-w"
                               else
                                   dt.preferences.write("gpsbabel", "selected_waypoints", "bool", false)
                                   gpsbabel_cmd_waypoints=""
                               end
                               
                               if (gpsbabel_check_button_tracks.value) then
                                   dt.preferences.write("gpsbabel", "selected_tracks", "bool", true)
                                   gpsbabel_cmd_tracks="-t"
                               else
                                   dt.preferences.write("gpsbabel", "selected_tracks", "bool", false)
                                   gpsbabel_cmd_tracks=""
                               end
                               if (gpsbabel_check_button_routes.value) then
                                   dt.preferences.write("gpsbabel", "selected_routes", "bool", true)
                                   gpsbabel_cmd_routes="-r"
                               else
                                   dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
                                   gpsbabel_cmd_routes=""
                               end
                               if (gpsbabel_check_button_interpolate.value) then
                                   dt.preferences.write("gpsbabel", "selected_interpolate", "bool", true)
                                   gpsbabel_cmd_interpolate="-x interpolate,time=1"
                               else
                                   dt.preferences.write("gpsbabel", "selected_interpolate", "bool", false)
                                   gpsbabel_cmd_interpolate=""
                               end
                                
                          -- create command
                               gpsbabelStartCommand="gpsbabel " ..gpsbabel_cmd_waypoints.. " " ..gpsbabel_cmd_tracks.. " " ..gpsbabel_cmd_routes.. " " ..gpsbabel_cmd_device..gpsbabel_cmd_erase_after_download.. " " ..gpsbabel_cmd_interface.. " " ..gpsbabel_cmd_outputformat.. " -F " ..gpsbabel_cmd_outputpath.. "/" ..gpsbabel_cmd_outputfile
                                  dt.print_error(gpsbabelStartCommand)    
                                  local result = nil
                                  dt.print(_('download started. please wait for the successfully download confirmation!'))
                                  gpsbabel_download_button.sensitive = false
                                  result = dt.control.execute( gpsbabelStartCommand)
                                  if (result == 0) then
                                     dt.print(_('download completed successfully'))
                                        if (gpsbabel_check_button_interpolate.value) then  -- interpolation
                                            dt.print(_('interpolate waypoints. Please wait for the successfully confirmation!'))
                                            local result2 = nil
                                             informat=  gpsbabel_combobox_output.value
                                             dt.print_error("gpsbabel -i " ..informat.. " -f "..gpsbabel_cmd_outputpath.. "/" ..gpsbabel_cmd_outputfile.. " -x interpolate,time=1 -o "..informat.." -F "..gpsbabel_cmd_outputpath.. "/" ..gpsbabel_cmd_outputfile)
                                            result2 = dt.control.execute("gpsbabel -i " ..informat.. " -f "..gpsbabel_cmd_outputpath.. "/" ..gpsbabel_cmd_outputfile.. " -x interpolate,time=1 -o "..informat.." -F "..gpsbabel_cmd_outputpath.. "/" ..gpsbabel_cmd_outputfile)
                                             if (result2 == 0) then    -- time zone correction
                                                  if (gpsbabel_combobox_timezone.value ~= "0h") then
                                                    local result3 = nil
                                                    timezone= gpsbabel_combobox_timezone.value
                                                    dt.print_error("gpsbabel -i " ..informat.. " -f "..gpsbabel_cmd_outputpath.. "/" ..gpsbabel_cmd_outputfile.. " -x track,move="..timezone.. " -o "..informat.." -F "..gpsbabel_cmd_outputpath.. "/" ..gpsbabel_cmd_outputfile)
                                                    result3 = dt.control.execute("gpsbabel -i " ..informat.. " -f "..gpsbabel_cmd_outputpath.. "/" ..gpsbabel_cmd_outputfile.. " -x track,move="..timezone.. " -o "..informat.." -F "..gpsbabel_cmd_outputpath.. "/" ..gpsbabel_cmd_outputfile)
                                                    if (result3 ==0) then
                                                    dt.print(_('time zone correction completed successfully'))
                                                    else
                                                    dt.print(_("ERROR: time zone correction doesn't work. For more information see terminal output"))
                                                    end
                                                  else
                                                  dt.print(_('download and interpolation completed successfully'))
                                                  end
                                             else
                                               dt.print(_("ERROR: interpolation doen't work. For more information see terminal output"))
                                             end
                                        elseif (gpsbabel_combobox_timezone.value ~= "0h") then -- only time zone correction
                                            timezone= gpsbabel_combobox_timezone.value
                                            informat=  gpsbabel_combobox_output.value
                                            dt.print_error("gpsbabel -i " ..informat.. " -f "..gpsbabel_cmd_outputpath.. "/" ..gpsbabel_cmd_outputfile.. " -x track,move="..timezone.. " -o "..informat.." -F "..gpsbabel_cmd_outputpath.. "/" ..gpsbabel_cmd_outputfile)
                                            result3 = dt.control.execute("gpsbabel -i " ..informat.. " -f "..gpsbabel_cmd_outputpath.. "/" ..gpsbabel_cmd_outputfile.. " -x track,move="..timezone.. " -o "..informat.." -F "..gpsbabel_cmd_outputpath.. "/" ..gpsbabel_cmd_outputfile)
                                            if (result3 ==0) then
                                              dt.print(_('time zone correction completed successfully'))
                                            else
                                              dt.print(_("ERROR: time zone correction doesn't work. For more information see terminal output"))
                                            end                
                                        end
                
                                     gpsbabel_download_button.sensitive = true
                                  else
                                     dt.print(_('ERROR: please check the device, interface and connection to the device. For more information see terminal output')) 
                                     gpsbabel_download_button.sensitive = true
                                  end
                          
                           else
                             dt.print(_('ERROR: no valid filename found. please enter a filename'))
                           end
                    else
                      dt.print(_('ERROR: no target directory selected. please select the target directory'))  
                     -- return
                    end
               else
                 dt.print(_('ERROR: no download option selected. please select one or more options'))
               --  return
               end
           else    
               dt.print(_('ERROR: gpsbabel not found. please install gpsbabel.'))
               dt.print_error(_('gpsbabel not found. please install gpsbabel.'))
             --  return
           end
      end
    }




    
widget = dt.new_widget("box") {
    orientation = "vertical",
    gpsbabel_label_source,
    gpsbabel_combobox_device,
    gpsbabel_combobox_interface,
    gpsbabel_file_chooser_input_button,
    gpsbabel_combobox_input,           
    gpsbabel_erase_only_button,
    gpsbabel_separator1,
    gpsbabel_label_download,
    gpsbabel_check_button_waypoints,
    gpsbabel_check_button_tracks,
    gpsbabel_check_button_routes,
    gpsbabel_check_button_delete_after,
    gpsbabel_separator2,
    gpsbabel_label_filter,
    gpsbabel_combobox_timezone,
    gpsbabel_check_button_interpolate, 
    gpsbabel_separator3,  
    gpsbabel_label_path,
    gpsbabel_combobox_output,
    gpsbabel_file_chooser_output_button,
    gpsbabel_label_filename,
    gpsbabel_entry_filename,
    gpsbabel_download_button,   
}


-- Register
dt.register_lib("gpsbabel",_('gps logfile downloader'), true, true,{[dt.gui.views.lighttable] = {"DT_UI_CONTAINER_PANEL_RIGHT_CENTER", 100}}, widget, nil, nil)


