--[[
    
    darktable is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    darktable is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with darktable.  If not, see <http://www.gnu.org/licenses/>.
]]
--[[
Version 2.2.1
* it creates a new GPS Logfile downloader module at the lighttable
* the module can download and convert logfiles
* the module is a frontend for gpsbabel (No complete support)
* supported devices:
   - Wintec WBT-100/200 GPS
   - Holux M-241 (MTK based)
   - Garmin serial/USB protocol 
   - GlobalSat DG-100/BT-335
   - GlobalSat DG-200
   - Brauniger IQ Series Barograph
   - MTK Logger (iBlue 747,Qstarz BT-1000,...
   - NaviGPS GT-11/BGT-11
   - SkyTraq Venus based loggers
* supported file fomats
   - GPX
   - CVS
   - KML
   - And many more. For more information see http://www.gpsbabel.org
   
DEPENDS
- gpsbabel version 1.5.4 or newer

                
USAGE
* 
- Select your device or file
- Select your interface or input file with file format 
- Select the download option, file format, directory and create a filename without suffix
- click download and wait for the download confirmation
                
                
]]

local dt = require "darktable"
local gettext = dt.gettext

-- works with darktable 2.2.X
dt.configuration.check_version(...,{4,0,0})

-- command variables
cmd_device=""
cmd_interface=""            --  Interface or File -f /dev/cu.WBT200-SPPslave-1 or -f /tmp/test.gpx
cmd_inputformat=""          --  Input device of fileformat -i gpx or -i wbt
cmd_outputformat=""         --  Outputfileformat  -o gpx
cmd_outputfile=""
cmd_outputsuffix=""
cmd_outputpath=""            
cmd_waypoints=""             --  Process waypoint information [default] -w
cmd_tracks=""                --  Process track information -t
cmd_routes=""                --  Process route information -r
cmd_erase_after_download=""
cmd_interpolate=""           --  filtername -x 
gpsbabelStartCommand=""

-- initialize combobox and checkbox value for default (Wintec WBT-100/200 GPS)
if dt.preferences.read("gpsbabel",  "selected_device", "integer") == 0 then

   dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)       --Interface is selectable
   dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     --Inputfile is not selectable 
   dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   --Inputfileformat is not selectable             
   dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)       --Checkbox not selectable
   dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)          --Checkbox not selectable
   dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)         --Checkbox not selectable
   dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", true)     --Checkbox not selectable
   dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false)     --Button not selectable
   dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true)     --Interpolate yes
    
   dt.preferences.write("gpsbabel", "selected_device", "integer", 1)           --Default device Wintec WBT-100/200 GPS
   dt.preferences.write("gpsbabel", "selected_interface", "integer", 2)        --Default interface /dev/ttyUSB0
   dt.preferences.write("gpsbabel", "selected_input_format", "integer", 1)     --Default input file format gpx            
   dt.preferences.write("gpsbabel", "selected_waypoints", "bool", true)        --Wintec WBT-100/200 GPS waypoints are supported
   dt.preferences.write("gpsbabel", "selected_tracks", "bool", true)           --Wintec WBT-100/200 GPS tracks are supported
   dt.preferences.write("gpsbabel", "selected_routes", "bool", false)          --Wintec WBT-100/200 GPS routes are nit spported   
   dt.preferences.write("gpsbabel", "selected_erase_after", "bool", true)      --Wintec WBT-100/200 GPS supports erase data after download
   dt.preferences.write("gpsbabel", "selected_interpolate", "bool", true)      --Interpolate wapoints   
   dt.preferences.write("gpsbabel", "selected_output_format", "integer", 1)    --GPX format

end

-- Tell gettext where to find the .mo file translating messages for a particular domain
gettext.bindtextdomain("gpsbabel",dt.configuration.config_dir.."/lua/")

local function _(msgid)
    return gettext.dgettext("gpsbabel", msgid)
end


-- check installation
local function checkIfBinExists(bin)
  local handle = io.popen("which "..bin)
  local result = handle:read()
  local ret
  handle:close()
  if (result) then
 --   dt.print_error("true checkIfBinExists: "..bin)
    ret = true
  else
    dt.print_error(bin.." not found")
    ret = false
  end


  return ret
end



-- start GUI
label_source = dt.new_widget("label")
{
     label = _('source:'),
     ellipsize = "start",
     halign = "start"
}

label_filename = dt.new_widget("label")
{
     label = _('filename without suffix:'),
     ellipsize = "start",
     halign = "start"
}
                
               
label_download = dt.new_widget("label")
{
     label = _('options:'),
     ellipsize = "start",
     halign = "start"
}                
    
label_path = dt.new_widget("label")
{
     label = _('target:'),
     ellipsize = "start",
     halign = "start"
}
                
label_filter = dt.new_widget("label")
{
     label = _('data filter:'),
     ellipsize = "start",
     halign = "start"
}
                
                
separator1 = dt.new_widget("separator")
{}

separator2 = dt.new_widget("separator")
{}                
                
separator3 = dt.new_widget("separator")
{}                  
                
    




check_button_waypoints = dt.new_widget("check_button")
{
    label = _('waypoints'), 
    value = dt.preferences.read("gpsbabel", "selected_waypoints", "bool"),
    sensitive = dt.preferences.read("gpsbabel", "sensitive_waypoints", "bool"),
    tooltip = _('download waypoints'),  
    clicked_callback = function(way)   
      --check device feature
      if (dt.preferences.read("gpsbabel", "sensitive_waypoints", "bool")) then
          if (way.value) then
               dt.preferences.write("gpsbabel", "selected_waypoints", "bool", true)
               cmd_waypoints="-w"
          --   return
          else
              dt.preferences.write("gpsbabel", "selected_waypoints", "bool", false)
              cmd_waypoints=""
         --    return
          end
      else
         check_button_waypoints.sensitive = false
         check_button_waypoints.value = false
         cmd_waypoints=""
      end
    end,
    
    reset_callback = function(self) 
       self.sensitive = true
       self.value = true
       dt.preferences.write("gpsbabel", "selected_waypoints", "bool", true)
       cmd_waypoints="-w"
    end
}
    

check_button_tracks = dt.new_widget("check_button")
{
    label = _('tracks'), 
    value = dt.preferences.read("gpsbabel", "selected_tracks", "bool"),
    sensitive = dt.preferences.read("gpsbabel", "sensitive_tracks", "bool"),
    tooltip = _('download tracks'),  
    clicked_callback = function(tra)  
       --check device feature
       if (dt.preferences.read("gpsbabel", "sensitive_tracks", "bool")) then
          if (tra.value) then
               dt.preferences.write("gpsbabel", "selected_tracks", "bool", true)
               cmd_tracks="-t"
             return
          else
              dt.preferences.write("gpsbabel", "selected_tracks", "bool", false)
              cmd_tracks=""
             return
          end
       else
         check_button_tracks.sensitive = false
         check_button_tracks.value = false
         cmd_tracks=""  
       end
    end,
    reset_callback = function(self) 
       self.sensitive = true
       self.value = true
       dt.preferences.write("gpsbabel", "selected_tracks", "bool", true)
       cmd_tracks="-t"
    end

}


check_button_routes = dt.new_widget("check_button")
{
    label = _('routes'), 
    value = dt.preferences.read("gpsbabel", "selected_routes", "bool"),
    sensitive = dt.preferences.read("gpsbabel", "sensitive_routes", "bool"),
    tooltip = _('download routes'),
    clicked_callback = function(rou)   
       --check device feature
       if (dt.preferences.read("gpsbabel", "sensitive_routes", "bool")) then
          if (rou.value) then
               dt.preferences.write("gpsbabel", "selected_routes", "bool", true)
               cmd_routes="-r"
             return
          else
              dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
              cmd_routes=""
             return
          end
       else
         check_button_routes.sensitive = false
         check_button_routes.value = false
         cmd_routes=""    
       end    
    end,
    reset_callback = function(self) 
       self.sensitive = false
       self.value = false
       dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
       cmd_routes=""
       dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
    end
}

check_button_delete_after = dt.new_widget("check_button")
{
    label = _('delete GPS logfiles after download'), 
    value = false,
    tooltip = _('delete the logfiles from the source after download'),
    sensitive = dt.preferences.read("gpsbabel", "sensitive_erase_after", "bool"),            
    reset_callback = function(self_delete_after) 
       self_delete_after.value = false
       self_delete_after.sensitive = dt.preferences.read("gpsbabel", "sensitive_erase_after", "bool")
    end,
    clicked_callback = function(execute_delete)
       --check device feature
       if (dt.preferences.read("gpsbabel", "sensitive_erase_after", "bool")) then
           if (execute_delete.value) then
             cmd_erase_after_download=",erase"
           else
             cmd_erase_after_download=""  
         end
       else
         check_button_delete_after.sensitive = false
         check_button_delete_after.value = false
         cmd_routes=""    
       end  
    end
}
    

check_button_interpolate = dt.new_widget("check_button")
{
    label = _('interpolate trackpoints'), 
    value = dt.preferences.read("gpsbabel", "selected_interpolate", "bool"),
    tooltip = _('this option adds every second a trackpoint between two logged trackpoints'),
    reset_callback = function(interself) 
       interself.value = true
       interself.sensitive = true
    end,
    clicked_callback = function(inter)
       --check device feature               
       if (dt.preferences.read("gpsbabel", "sensitive_interpolate", "bool")) then
           if (inter.value) then
             dt.preferences.write("gpsbabel", "selected_interpolate", "bool", true)
             cmd_interpolate="interpolate,time=1"
           else
             dt.preferences.write("gpsbabel", "selected_interpolate", "bool", false)
             cmd_interpolate=""  
         end
       else
         inter.sensitive = false
         inter.value = false
         cmd_interpolate=""    
       end  
    end
}                
                            

file_chooser_input_button = dt.new_widget("file_chooser_button")
{
    title = _('source file'),  -- The title of the window when choosing a file
    value = "",                       -- The currently selected file
    tooltip = _('please select your source file'),
    is_directory = false,              -- True if the file chooser button only allows directories to be selecte
    sensitive = dt.preferences.read("gpsbabel", "sensitive_input_file", "bool"),
    changed_callback = function(input_path)             
      cmd_interface = "-f "..input_path.value 
   --   combobox_input.value = dt.preferences.read("gpsbabel", "selected_input_format", "integer")   PRÜFEN OB NOTWENDIG
    end
                
}                
                

                
erase_only_button = dt.new_widget("button")
    {
      label = _('delete source data without download'),
      sensitive = dt.preferences.read("gpsbabel", "sensitive_erase_only", "bool"),
      clicked_callback = function (_)
         -- check device feature
         if (dt.preferences.read("gpsbabel", "sensitive_erase_only", "bool") and combobox_interface.sensitive) then
              -- check gpsbabel installation
              if (checkIfBinExists("gpsbabel")) then
               -- create command
                   cmd_interface = "-f " ..combobox_interface.value
                   gpsbabelStartCommand="gpsbabel " ..cmd_device..",erase_only " ..cmd_interface
                   dt.print_error(gpsbabelStartCommand)    
                   local result = nil
                   dt.print(_('erase log data on the device...'))
                   result = dt.control.execute( gpsbabelStartCommand)
                    if (result == 0) then
                     dt.print(_('erase log data completed successfully'))
                   else
                     dt.print(_('ERROR: please check the device, interface and connection to the device. For more information see terminal output')) 
                   end
              else    
                  dt.print(_('ERROR: gpsbabel not found. Please install gpsbabel.'))
                  dt.print_error(_('gpsbabel not found. Please install gpsbabel.'))
                --  return
              end
         else
           erase_only_button.sensitive = false
         end
      end
    }                
                
                
                



file_chooser_output_button = dt.new_widget("file_chooser_button")
{
    title = _('download GPS data'),  -- The title of the window when choosing a file
    is_directory = true,  -- True if the file chooser button only allows directories to be selecte
    tooltip = _('select the directory for the target file'),
    changed_callback = function(path)
       dt.preferences.write("gpsbabel", "path", "string", file_chooser_output_button.value)
       cmd_outputpath= file_chooser_output_button.value.."/"
    end

}



entry_filename = dt.new_widget("entry")
{
    text = "$(YEAR)-$(MONTH)-$(DAY)", 
    sensitive = true,
    is_password = true,
    editable = true,
    tooltip = _('enter the filename without suffix. the filename $(YEAR)-$(MONTH)-$(DAY) creates a file like 2017-02-22.gpx'),
    reset_callback = function(self) 
       self.text = "$(YEAR)-$(MONTH)-$(DAY)" 
    end

}

combobox_timezone = dt.new_widget("combobox")
{
    label = _('rectify time zone'), 
    tooltip = _('this option can rectify the time zone from your GPS device'),
    editable = false,
    value = 24, "-23h","-22h","-21h","-20h","-19h","-18h","-17h","-16h","-15h","-14h","-13h","-12h","-11h","-10h","-9h","-8h","-7h","-6h","-5h","-4h","-3h","-2h","-1h","0h","+1h","+2h","+3h","+4h","+5h","+6h","+7h","+8h","+9h","+10h","+11h","+12h","+13h","+14h","+15h","+16h","+17h","+18h","+19h","+20h","+21h","+22h","+23h",
    reset_callback = function(reset_timezone) 
       reset_timezone.sensitive = true
       reset_timezone.value = 24
    end, 
                
}                
                
                
combobox_output = dt.new_widget("combobox")
{
    label = _('file format'), 
    tooltip = _('please select the output format'),
    editable = false,
    value = dt.preferences.read("gpsbabel", "selected_output_format", "integer"),
    changed_callback = function(output_path) 
       cmd_outputformat = "-o "..output_path.value
       cmd_outputsuffix = "."..output_path.value
       dt.preferences.write("gpsbabel", "selected_output_format", "integer", output_path.selected)
    end,
    "gpx", "kml","csv","html",
    
    reset_callback = function(reset) 
       reset.value = 1
       cmd_outputformat = "-o gpx" 
    end,
}
                
combobox_interface = dt.new_widget("combobox")
{
    label = _('interface'), 
    tooltip = _('please selecting your interface'),
    editable = true,
    sensitive = dt.preferences.read("gpsbabel", "sensitive_interface", "bool"),           
    value = dt.preferences.read("gpsbabel", "selected_interface", "integer"),
    changed_callback = function(selinter) 
      cmd_interface = "-f "..selinter.value
      dt.preferences.write("gpsbabel", "selected_interface", "integer", selinter.selected)
    end,
    "usb:", "/dev/ttyUSB0", "com1:","/dev/cu.WBT200-SPPslave-1","/dev/rfcomm0","/dev/ttyS0",
    
    reset_callback = function(interfacereset) 
       interfacereset.value = 2
       cmd_interface = "-f " ..combobox_interface.value
    end,   
}


function setsensitive()
 
    combobox_interface.sensitive = dt.preferences.read("gpsbabel", "sensitive_interface", "bool")    
    file_chooser_input_button.sensitive = dt.preferences.read("gpsbabel", "sensitive_input_file", "bool")
    combobox_input.sensitive = dt.preferences.read("gpsbabel", "sensitive_input_format", "bool")
    erase_only_button.sensitive = dt.preferences.read("gpsbabel", "sensitive_erase_only", "bool") 
    check_button_waypoints.sensitive = dt.preferences.read("gpsbabel", "sensitive_waypoints", "bool")
    check_button_tracks.sensitive =  dt.preferences.read("gpsbabel", "sensitive_tracks", "bool")
    check_button_routes.sensitive = dt.preferences.read("gpsbabel", "sensitive_routes", "bool")
    check_button_delete_after.sensitive = dt.preferences.read("gpsbabel", "sensitive_erase_after", "bool") 
                
    if  (check_button_waypoints.sensitive) then           
      check_button_waypoints.value = true
    else
      check_button_waypoints.value = false
    end
    
    if  (check_button_tracks.sensitive) then           
      check_button_tracks.value = true
    else
      check_button_tracks.value = false
    end            
                
    if  (check_button_routes.sensitive) then           
      check_button_routes.value = true
    else
      check_button_routes.value = false
    end              
                
    if  (check_button_delete_after.sensitive) then           
      check_button_delete_after.value = true
    else
      check_button_delete_after.value = false
    end              
                
end                               
                
                
 combobox_input = dt.new_widget("combobox")
{
    label = _('file format'), 
    tooltip = _('please select the input format. more informations see gpsbabel homepage'),
    editable = false,
    sensitive = dt.preferences.read("gpsbabel", "sensitive_input_format", "bool"),
    value = dt.preferences.read("gpsbabel", "selected_input_format", "integer"),
    changed_callback = function(input) 
        if (input.value == "GPX exchange format (gpx)") then
          cmd_device = "-i gpx"
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()     
                
        elseif (input.value == "Keyhole Markup Language (kml)") then   
          cmd_device = "-i kml"
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()  
        elseif (input.value == "Comma-separated values (csv)") then                  
          cmd_device = "-i csv" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()  
         elseif (input.value == "Universal csv with field structure (csv)") then                  
          cmd_device = "-i unicsv" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()  
        elseif (input.value == "Alan Map500 tracklogs") then                  
          cmd_device = "-i alantrl" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
        elseif (input.value == "Alan Map500 waypoints and routes") then                  
          cmd_device = "-i alanwpr" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "Bushnell GPS Trail file") then                  
          cmd_device = "-i bushnell_trl" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
         elseif (input.value == "Bushnell GPS Waypoint file") then                  
          cmd_device = "-i bushnell" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
        elseif (input.value == "Cambridge/Winpilot glider software") then                  
          cmd_device = "-i cambridge" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
        elseif (input.value == "CarteSurTable data file") then                  
          cmd_device = "-i cst" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
         elseif (input.value == "Columbus/Visiontac V900 files (.csv)") then                  
          cmd_device = "-i v900" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
         elseif (input.value == "CompeGPS data files (.wpt/.trk/.rte)") then                  
          cmd_device = "-i compegps" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
         elseif (input.value == "Data Logger iBlue747 csv") then                  
          cmd_device = "-i iblue747" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
        elseif (input.value == "Data Logger iBlue757 csv") then                  
          cmd_device = "-i iblue757" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()       
        elseif (input.value == "DeLorme .an1 (drawing) file") then                  
          cmd_device = "-i an1" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
        elseif (input.value == "DeLorme GPL") then                  
          cmd_device = "-i gpl" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()     
        elseif (input.value == "DeLorme Street Atlas Plus") then                  
          cmd_device = "-i saplus" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "DeLorme Street Atlas Route") then                  
          cmd_device = "-i saroute" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()   
        elseif (input.value == "DeLorme XMap HH Native .WPT") then                  
          cmd_device = "-i xmap" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
        elseif (input.value == "DeLorme XMap/SAHH 2006 Native .TXT") then                  
          cmd_device = "-i xmap2006" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "DeLorme XMat HH Street Atlas USA .WPT (PPC)") then                  
          cmd_device = "-i xmapwpt" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()     
        elseif (input.value == "Destinator Itineraries (.dat)") then                  
          cmd_device = "-i destinator_itn" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
         elseif (input.value == "Destinator Points of Interest (.dat)") then                  
          cmd_device = "-i destinator_poi" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()  
        elseif (input.value == "Destinator TrackLogs (.dat)") then                  
          cmd_device = "-i destinator_trl" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()  
        elseif (input.value == "EasyGPS binary format") then                  
          cmd_device = "-i easygps" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
        elseif (input.value == "Embedded Exif-GPS data (.jpg)") then                  
          cmd_device = "-i exif" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "Energympro GPS training watch") then                  
          cmd_device = "-i energympro" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
        elseif (input.value == "Flexible and Interoperable Data Transfer (FIT) Activity file") then                  
          cmd_device = "-i garmin_fit" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive() 
        elseif (input.value == "Fugawi") then                  
          cmd_device = "-i fugawi" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
        elseif (input.value == "G7ToWin data files (.g7t)") then                  
          cmd_device = "-i g7towin" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "Garmin 301 Custom position and heartrate") then                  
          cmd_device = "-i garmin301" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
       elseif (input.value == "Garmin G1000 datalog input filter file") then                  
          cmd_device = "-i garmin_g1000" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()            
        elseif (input.value == "Garmin Logbook XML") then                  
          cmd_device = "-i glogbook" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()           
         elseif (input.value == "Garmin MapSource - gdb") then                  
          cmd_device = "-i gdb" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
         elseif (input.value == "Garmin MapSource - mps") then                  
          cmd_device = "-i mapsource" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()  
        elseif (input.value == "Garmin MapSource - txt (tab delimited)") then                  
          cmd_device = "-i garmin_txt" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()      
         elseif (input.value == "Garmin PCX5") then                  
          cmd_device = "-i pcx" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "Garmin POI database") then                  
          cmd_device = "-i garmin_poi" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
         elseif (input.value == "Garmin Points of Interest (.gpi)") then                  
          cmd_device = "-i garmin_gpi" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
        elseif (input.value == "Garmin Training Center (.tcx,.tcx)") then                  
          cmd_device = "-i gtrnctr" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()
       elseif (input.value == "Geocaching.com .loc") then                  
          cmd_device = "-i geo" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()            
        elseif (input.value == "GlobalSat GH625XT GPS training watch") then                  
          cmd_device = "-i globalsat" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()       
        elseif (input.value == "Google Directions XML") then                  
          cmd_device = "-i googledir" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
         elseif (input.value == "Google Navigator Tracklines (.trl)") then                  
          cmd_device = "-i gnav_trl" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
         elseif (input.value == "GoPal GPS track log (.trk)") then                  
          cmd_device = "-i gopal" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
         elseif (input.value == "GPS Tracking Key Pro text") then                  
          cmd_device = "-i land_air_sea" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()   
         elseif (input.value == "GPS TrackMaker") then                  
          cmd_device = "-i gtm" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
         elseif (input.value == "GPSman") then                  
          cmd_device = "-i gpsman" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive() 
         elseif (input.value == "gpsutil") then                  
          cmd_device = "-i gpsutil" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
        elseif (input.value == "HikeTech") then                  
          cmd_device = "-i hiketech" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
        elseif (input.value == "Holux (gm-100) .wpo Format") then                  
          cmd_device = "-i holux" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive() 
        elseif (input.value == "Holux M-241 (MTK based) Binary File Format") then                  
          cmd_device = "-i m241-bin" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "Jelbert GeoTagger data file") then                  
          cmd_device = "-i jtr" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
         elseif (input.value == "Lowrance USR") then                  
          cmd_device = "-i lowranceusr" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
         elseif (input.value == "Lowrance USR version 4") then                  
          cmd_device = "-i lowranceusr4" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()  
        elseif (input.value == "Magellan Explorist Geocaching") then                  
          cmd_device = "-i maggeo" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
        elseif (input.value == "Magellan SD files (as for eXplorist)") then                  
          cmd_device = "-i magellanx" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
         elseif (input.value == "Magellan SD files (as for Meridian)") then                  
          cmd_device = "-i magellan" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "Magellan serial protocol") then                  
          cmd_device = "-i magellan" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()       
        elseif (input.value == "Mainnav") then                  
          cmd_device = "-i mainnav" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
        elseif (input.value == "Navigon Mobile Navigator .rte files") then                  
          cmd_device = "-i nmn4" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
         elseif (input.value == "Navigon Waypoints") then                  
          cmd_device = "-i navigonwpt" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
         elseif (input.value == "OpenStreetMap data files") then                  
          cmd_device = "-i osm" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()         
         elseif (input.value == "Wintec TES file") then                  
          cmd_device = "-i wintec_tes" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()   
         elseif (input.value == "Wintec WBT-100/200 Binary File Format") then                  
          cmd_device = "-i wbt-bin" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()          
         elseif (input.value == "Wintec WBT-201/G-Rays 2 Binary File Format (tk1)") then                  
          cmd_device = "-i wbt-tk1" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()        
         elseif (input.value == "XAiOX iTrackU Logger") then                  
          cmd_device = "-i itracku" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()    
        elseif (input.value == "Yahoo Geocode API data") then                  
          cmd_device = "-i yahoo" 
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          setsensitive()       
        end        

       dt.preferences.write("gpsbabel", "selected_input_format", "integer", input.selected)
    end,
    "GPX exchange format (gpx)", "Keyhole Markup Language (kml)","Comma-separated values (csv)","Universal csv with field structure (csv)","Alan Map500 tracklogs","Alan Map500 waypoints and routes","Bushnell GPS Trail file","Bushnell GPS Waypoint file","Cambridge/Winpilot glider software","CarteSurTable data file","Columbus/Visiontac V900 files (.csv)","CompeGPS data files (.wpt/.trk/.rte)","Data Logger iBlue747 csv","Data Logger iBlue757 csv","DeLorme .an1 (drawing) file","DeLorme GPL","DeLorme Street Atlas Plus","DeLorme Street Atlas Route","DeLorme XMap HH Native .WPT","DeLorme XMap/SAHH 2006 Native .TXT","DeLorme XMat HH Street Atlas USA .WPT (PPC)","Destinator Itineraries (.dat)","Destinator Points of Interest (.dat)","Destinator TrackLogs (.dat)","EasyGPS binary format","Embedded Exif-GPS data (.jpg)","Energympro GPS training watch","Flexible and Interoperable Data Transfer (FIT) Activity file","Fugawi","G7ToWin data files (.g7t)","Garmin 301 Custom position and heartrate","Garmin G1000 datalog input filter file","Garmin Logbook XML","Garmin MapSource - gdb","Garmin MapSource - mps","Garmin MapSource - txt (tab delimited)","Garmin PCX5","Garmin POI database","Garmin Points of Interest (.gpi)","Garmin Training Center (.tcx)","Garmin Training Center (.xml,.tcx)","Geocaching.com .loc","GlobalSat GH625XT GPS training watch","Google Directions XML","Google Navigator Tracklines (.trl)","GoPal GPS track log (.trk)","GPS Tracking Key Pro text","GPS TrackMaker","GPSman","gpsutil","HikeTech","Holux (gm-100) .wpo Format","Holux M-241 (MTK based) Binary File Format","Jelbert GeoTagger data file","Lowrance USR","Lowrance USR version 4","Magellan Explorist Geocaching","Magellan SD files (as for eXplorist)","Magellan SD files (as for Meridian)","Magellan serial protocol","Mainnav","Navigon Mobile Navigator .rte files","Navigon Waypoints","OpenStreetMap data files","Wintec TES file","Wintec WBT-100/200 Binary File Format","Wintec WBT-201/G-Rays 2 Binary File Format (tk1)","XAiOX iTrackU Logger","Yahoo Geocode API data",
                
    reset_callback = function(reset_input) 
       reset_input.value = 1
       cmd_inputdevice = "-i gpx" 
    end,
}               
                
combobox_device = dt.new_widget("combobox")
{
    label = _('device or file'), 
    tooltip = _('please select your device or choose file'),
    value = dt.preferences.read("gpsbabel", "selected_device", "integer"),
    changed_callback = function(sd) 
       dt.preferences.write("gpsbabel", "selected_device", "integer", sd.selected) --sd.selected ist eine integer Zahl
       if (sd.value == "Wintec WBT-100/200 GPS") then
          cmd_device="-i wbt"
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
                    
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false)   
          setsensitive()  
          
       elseif  (sd.value == "Holux M-241 (MTK based)") then  
          cmd_device="-i m241"
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          
          dt.preferences.write("gpsbabel", "selected_waypointes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false) 
          setsensitive() 
    
        elseif  (sd.value == "Garmin serial/USB protocol") then  
          cmd_device="-i garmin"
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false) 
          setsensitive() 
          
        elseif  (sd.value == "GlobalSat DG-100/BT-335") then  
          cmd_device="-i dg-100"
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false) 
          setsensitive()  
          
        elseif  (sd.value == "GlobalSat DG-200") then  
          cmd_device="-i dg-200" 
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false) 
          setsensitive()  
          
        elseif  (sd.value == "Brauniger IQ Series Barograph") then 
          cmd_device="-i baroiq"
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
                    
          dt.preferences.write("gpsbabel", "selected_waypoints", "bool", false)
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false)
          setsensitive()
          
        elseif  (sd.value == "MTK Logger (iBlue 747,Qstarz BT-1000,...)") then
          cmd_device="-i mtk"   
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false) 
          setsensitive()   
          
           
        elseif  (sd.value == "NaviGPS GT-11/BGT-11") then   
          cmd_device="-i navilink"
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true)  
          
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false) 
          setsensitive()  
          
        elseif  (sd.value == "SkyTraq Venus based loggers") then
          cmd_device="-i skytraq"
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", false)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", false)   
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", true) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false)
          setsensitive()  
        
        elseif  (sd.value == "File") then
          cmd_device="-i "..dt.preferences.read("gpsbabel", "selected_input_format", "integer")
          dt.preferences.write("gpsbabel", "sensitive_interface", "bool", false)
          dt.preferences.write("gpsbabel", "sensitive_input_file", "bool", true)     
          dt.preferences.write("gpsbabel", "sensitive_input_format", "bool", true)                       
          dt.preferences.write("gpsbabel", "sensitive_waypoints", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_tracks", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_routes", "bool", true)
          dt.preferences.write("gpsbabel", "sensitive_erase_after", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_erase_only", "bool", false) 
          dt.preferences.write("gpsbabel", "sensitive_interpolate", "bool", true) 
          
          dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
          dt.preferences.write("gpsbabel", "selected_erase_after", "bool", false)       
           
           --set sensitive (sen_interface,sen_inputfile,sen_inputformat,sen_eraseonly,sen_waypoints,sen_tracks,sen_routes,sen_deleteafter)         
          setsensitive()     
       end

    end,
    "Wintec WBT-100/200 GPS", "Holux M-241 (MTK based)", "Garmin serial/USB protocol", "GlobalSat DG-100/BT-335", "GlobalSat DG-200", "Brauniger IQ Series Barograph", "MTK Logger (iBlue 747,Qstarz BT-1000,...)", "NaviGPS GT-11/BGT-11", "SkyTraq Venus based loggers", "File",
   reset_callback = function(device_reset) 
       device_reset.value = 1  --Default device Wintec WBT-100/200 GPS
    end,
 

}
                         
                
download_button = dt.new_widget("button")
    {
      label = _('process GPS logfile'),
      clicked_callback = function (_)
            -- check gpsbabel installation
           if (checkIfBinExists("gpsbabel")) then
               -- check download options
               if ((check_button_waypoints.value) or (check_button_tracks.value) or (check_button_routes.value)) then
                    -- check path
                    if (file_chooser_output_button.value ~= nil) then
                        cmd_outputpath = file_chooser_output_button.value
                         -- check filename
                           if (not (entry_filename.text == "")) then
                              if (entry_filename.text == "$(YEAR)-$(MONTH)-$(DAY)") then
                                 local YYYY = tostring(os.date("%Y"))
                                 local MM = tostring(os.date("%m"))
                                 local DD = tostring(os.date("%d"))
                                 cmd_outputfile = YYYY.. "-" ..MM.. "-" ..DD..cmd_outputsuffix
                               else
                                cmd_outputfile = entry_filename.text..cmd_outputsuffix
                               end
                               
                               -- check options and filters
                               if (check_button_waypoints.value) then
                                 dt.preferences.write("gpsbabel", "selected_waypoints", "bool", true)
                                 cmd_waypoints="-w"
                               else
                                   dt.preferences.write("gpsbabel", "selected_waypoints", "bool", false)
                                   cmd_waypoints=""
                               end
                               
                               if (check_button_tracks.value) then
                                   dt.preferences.write("gpsbabel", "selected_tracks", "bool", true)
                                   cmd_tracks="-t"
                               else
                                   dt.preferences.write("gpsbabel", "selected_tracks", "bool", false)
                                   cmd_tracks=""
                               end
                               if (check_button_routes.value) then
                                   dt.preferences.write("gpsbabel", "selected_routes", "bool", true)
                                   cmd_routes="-r"
                               else
                                   dt.preferences.write("gpsbabel", "selected_routes", "bool", false)
                                   cmd_routes=""
                               end
                               if (check_button_interpolate.value) then
                                   dt.preferences.write("gpsbabel", "selected_interpolate", "bool", true)
                                   cmd_interpolate="-x interpolate,time=1"
                               else
                                   dt.preferences.write("gpsbabel", "selected_interpolate", "bool", false)
                                   cmd_interpolate=""
                               end
                                
                          -- create command
                               gpsbabelStartCommand="gpsbabel " ..cmd_waypoints.. " " ..cmd_tracks.. " " ..cmd_routes.. " " ..cmd_device..cmd_erase_after_download.. " " ..cmd_interface.. " " ..cmd_outputformat.. " -F " ..cmd_outputpath.. "/" ..cmd_outputfile
                                  dt.print_error(gpsbabelStartCommand)    
                                  local result = nil
                                  dt.print(_('download started. please wait for the successfully download confirmation!'))
                                  download_button.sensitive = false
                                  result = dt.control.execute( gpsbabelStartCommand)
                                  if (result == 0) then
                                     dt.print(_('download completed successfully'))
                                        if (check_button_interpolate.value) then  -- interpolation
                                            dt.print(_('interpolate waypoints. Please wait for the successfully confirmation!'))
                                            local result2 = nil
                                             informat=  combobox_output.value
                                             dt.print_error("gpsbabel -i " ..informat.. " -f "..cmd_outputpath.. "/" ..cmd_outputfile.. " -x interpolate,time=1 -o "..informat.." -F "..cmd_outputpath.. "/" ..cmd_outputfile)
                                            result2 = dt.control.execute("gpsbabel -i " ..informat.. " -f "..cmd_outputpath.. "/" ..cmd_outputfile.. " -x interpolate,time=1 -o "..informat.." -F "..cmd_outputpath.. "/" ..cmd_outputfile)
                                             if (result2 == 0) then    -- time zone correction
                                                  if (combobox_timezone.value ~= "0h") then
                                                    local result3 = nil
                                                    timezone= combobox_timezone.value
                                                    dt.print_error("gpsbabel -i " ..informat.. " -f "..cmd_outputpath.. "/" ..cmd_outputfile.. " -x track,move="..timezone.. " -o "..informat.." -F "..cmd_outputpath.. "/" ..cmd_outputfile)
                                                    result3 = dt.control.execute("gpsbabel -i " ..informat.. " -f "..cmd_outputpath.. "/" ..cmd_outputfile.. " -x track,move="..timezone.. " -o "..informat.." -F "..cmd_outputpath.. "/" ..cmd_outputfile)
                                                    if (result3 ==0) then
                                                    dt.print(_('time zone correction completed successfully'))
                                                    else
                                                    dt.print(_("ERROR: time zone correction doesn't work. For more information see terminal output"))
                                                    end
                                                  else
                                                  dt.print(_('download and interpolation completed successfully'))
                                                  end
                                             else
                                               dt.print(_("ERROR: interpolation doen't work. For more information see terminal output"))
                                             end
                                        elseif (combobox_timezone.value ~= "0h") then -- only time zone correction
                                            timezone= combobox_timezone.value
                                            informat=  combobox_output.value
                                            dt.print_error("gpsbabel -i " ..informat.. " -f "..cmd_outputpath.. "/" ..cmd_outputfile.. " -x track,move="..timezone.. " -o "..informat.." -F "..cmd_outputpath.. "/" ..cmd_outputfile)
                                            result3 = dt.control.execute("gpsbabel -i " ..informat.. " -f "..cmd_outputpath.. "/" ..cmd_outputfile.. " -x track,move="..timezone.. " -o "..informat.." -F "..cmd_outputpath.. "/" ..cmd_outputfile)
                                            if (result3 ==0) then
                                              dt.print(_('time zone correction completed successfully'))
                                            else
                                              dt.print(_("ERROR: time zone correction doesn't work. For more information see terminal output"))
                                            end                
                                        end
                
                                     download_button.sensitive = true
                                  else
                                     dt.print(_('ERROR: please check the device, interface and connection to the device. For more information see terminal output')) 
                                     download_button.sensitive = true
                                  end
                          
                           else
                             dt.print(_('ERROR: no valid filename found. please enter a filename'))
                           end
                    else
                      dt.print(_('ERROR: no target directory selected. please select the target directory'))  
                     -- return
                    end
               else
                 dt.print(_('ERROR: no download option selected. please select one or more options'))
               --  return
               end
           else    
               dt.print(_('ERROR: gpsbabel not found. please install gpsbabel.'))
               dt.print_error(_('gpsbabel not found. please install gpsbabel.'))
             --  return
           end
      end
    }




    
widget = dt.new_widget("box") {
    orientation = "vertical",
    label_source,
    combobox_device,
    combobox_interface,
    file_chooser_input_button,
    combobox_input,           
    erase_only_button,
    separator1,
    label_download,
    check_button_waypoints,
    check_button_tracks,
    check_button_routes,
    check_button_delete_after,
    separator2,
    label_filter,
    combobox_timezone,
    check_button_interpolate, 
    separator3,  
    label_path,
    combobox_output,
    file_chooser_output_button,
    label_filename,
    entry_filename,
    download_button,   
}


-- Register
dt.register_lib("gpsbabel",_('gps logfile downloader'), true, true,{[dt.gui.views.lighttable] = {"DT_UI_CONTAINER_PANEL_RIGHT_CENTER", 100}}, widget, nil, nil)


