--[[
  contact print plugin for darktable 2.2.X 

  copyright (c) 2017  Holger Klemm
  
  darktable is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.
  
  darktable is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with darktable.  If not, see <http://www.gnu.org/licenses/>.
]]

--[[
Version 2.1.2

ADDITIONAL SOFTWARE NEEDED FOR THIS SCRIPT
* montage (from ImageMagick)
* mogrify (from ImageMagick)
* exiftool

Change report:
* fix compatibility with other scripts
   
USAGE
* require this file from your main luarc config file.

This plugin will add a new storage option and calls hugin after export.
]]

local dt = require "darktable"
local gettext = dt.gettext

-- works only with darktable API version 4.0.0
dt.configuration.check_version(...,{4,0,0})

-- Tell gettext where to find the .mo file translating messages for a particular domain
gettext.bindtextdomain("contact_print",dt.configuration.config_dir.."/lua/")

local function _(msgid)
    return gettext.dgettext("contact_print", msgid)
end

-- command variables
contact_print_cmd_title=""
contact_print_cmd_output_path=""
contact_print_contact_print_cmd_output_filename=""
contact_print_cmd_suffix_output_format=""
contact_print_cmd_border=""
contact_print_cmd_print_filename=""
contact_print_cmd_images_per_line=""   
contact_print_cmd_lines_per_print=""  
contact_print_cmd_column_width=""
contact_print_cmd_line_width=""
contact_print_cmd_font_size=""
contact_print_cmd_input_files=""
contact_print_cmd_output_file=""
contact_print_montageStartCommand=""

-- initialize combobox and checkbox value for default
if dt.preferences.read("contact_print",  "images_per_line", "integer") == 0 then
    -- defaults
   dt.preferences.write("contact_print", "images_per_line", "integer", 5)  --5 images 
   dt.preferences.write("contact_print", "lines_per_print", "integer", 1)  --auto lines  
   dt.preferences.write("contact_print", "column_width", "integer", 5)     --5 pixel vertical gap
   dt.preferences.write("contact_print", "line_width", "integer", 5)       --5 pixel horizontal gap
   dt.preferences.write("contact_print", "border", "bool", false)          --without border
   dt.preferences.write("contact_print", "filename", "bool", true)         --with filename
   dt.preferences.write("contact_print", "font_size", "integer", 4)        --font size 32
   dt.preferences.write("contact_print", "output_format", "integer", 2)    --JPEG ou   
end



local contact_print_label_options= dt.new_widget("label")
{
     label = _('contact print options'),
     ellipsize = "end",
     halign = "end"
}


local contact_print_label_title= dt.new_widget("label")
{
     label = _('title'),
     ellipsize = "start",
     halign = "start"
}


local contact_print_entry_title = dt.new_widget("entry")
{
    text = "", 
    is_password = true,
    editable = true,
    tooltip = _('enter the contact print title'),
    reset_callback = function(self) 
       self.text = "" 
    end

}

contact_print_combobox_images_per_line = dt.new_widget("combobox")
{
    label = _('images per line'), 
    tooltip =_(''),
    value =  dt.preferences.read("contact_print", "images_per_line", "integer"), 
    changed_callback = function(sel_images_per_line) 
    dt.preferences.write("contact_print", "images_per_line", "integer", sel_images_per_line.selected)
    end,
    "1", "2", "3","4","5","6","7","8","9","10","11","12","13","14","15","16","17","18","19","20",
    reset_callback = function(self_images_line)
       self_images_line.value = 5
    end
} 

contact_print_combobox_lines_per_print = dt.new_widget("combobox")
{
    label = _('image lines'), 
    tooltip =_(''),
    value =  dt.preferences.read("contact_print", "lines_per_print", "integer"), 
    changed_callback = function(sel_lines_per_print) 
     dt.preferences.write("contact_print", "lines_per_print", "integer", sel_lines_per_print.selected)
    end,
    "auto","1", "2", "3","4","5","6","7","8","9","10","11","12","13","14","15","16","17","18","19","20",
    reset_callback = function(self_print)
       self_print.value = 1
    end
} 

contact_print_combobox_column_width = dt.new_widget("combobox")
{
    label = _('vertical gap between the images'), 
    tooltip =_(''),
    value =  dt.preferences.read("contact_print", "column_width", "integer"), 
    changed_callback = function(sel_column_width) 
      dt.preferences.write("contact_print", "column_width", "integer", sel_column_width.selected)
    end,
    "1", "2", "3","4","5","6","7","8","9","10","11","12","13","14","15","16","17","18","19","20",
    
    reset_callback = function(self_grid)
       self_grid.value = 5
    end
} 

contact_print_combobox_line_width = dt.new_widget("combobox")
{
    label = _('horizontal gap between the images'), 
    tooltip =_('creates a gap between '),
    value =  dt.preferences.read("contact_print", "line_width", "integer"), 
    changed_callback = function(sel_line_width) 
      dt.preferences.write("contact_print", "line_width", "integer", sel_line_width.selected)
    end,
    "1", "2", "3","4","5","6","7","8","9","10","11","12","13","14","15","16","17","18","19","20",
    
    reset_callback = function(self_grid)
       self_grid.value = 5
    end
} 


contact_print_check_button_rotate = dt.new_widget("check_button")
{
    label = _('rotate portrait orientation image'), 
    value = dt.preferences.read("contact_print", "rotate", "bool"), 
    tooltip =_(''),  
    clicked_callback = function(checked_rotate)   
          if (checked_rotate.value) then
               dt.preferences.write("contact_print", "rotate", "bool", true)
          else
              dt.preferences.write("contact_print", "rotate", "bool", false)
          end
    end,
    reset_callback = function(self_rotate) 
       self_rotate.value = false
    end
    
}

contact_print_check_button_border = dt.new_widget("check_button")
{
    label = _('images with black border'), 
    value = dt.preferences.read("contact_print", "border", "bool"), 
    tooltip =_(''),  
    clicked_callback = function(checked_border)   
          if (checked_border.value) then
               dt.preferences.write("contact_print", "border", "bool", true)
          else
              dt.preferences.write("contact_print", "border", "bool", false)
          end
    end,
    reset_callback = function(self_border) 
       self_border.value = false
    end
    
}



contact_print_check_button_print_filename = dt.new_widget("check_button")
{
    label = _('images with filename'), 
    value = dt.preferences.read("contact_print", "filename", "bool"), 
    tooltip =_(''), 
    clicked_callback = function(checked_filename)   
          if (checked_filename.value) then
               dt.preferences.write("contact_print", "filename", "bool", true)
          else
              dt.preferences.write("contact_print", "filename", "bool", false)
          end
    end,
    reset_callback = function(self) 
       self.value = true
    end
}

contact_print_combobox_font_size = dt.new_widget("combobox")
{
    label = _('filename font size'), 
    tooltip =_(''),
    value =  dt.preferences.read("contact_print", "font_size", "integer"), 
    changed_callback = function(sel_font_size) 
      dt.preferences.write("contact_print", "font_size", "integer", sel_font_size.selected)
    end,
    "14", "20", "26","32","40","48","60","72","88","96",
    
    reset_callback = function(self_font_size)
       self_font_size.value = 4
    end
} 



local contact_print_separator1 = dt.new_widget("separator")
{
}

local label_target= dt.new_widget("label")
{
     label = _('target image'),
     ellipsize = "start",
     halign = "end"
}

contact_print_combobox_output_format = dt.new_widget("combobox")
{
    label = _('file format'), 
    value = dt.preferences.read("contact_print", "output_format", "integer"), --2, "TIFF", "JPEG", "PNG","PNM","PBM","PGM","PPM",
    changed_callback = function(sel_output_format) 
      dt.preferences.write("contact_print", "output_format", "integer", sel_output_format.selected)
    end,
    "TIFF", "JPEG", "PNG","PNM","PBM","PPM",            
    reset_callback = function(self_output_format)
       self_output_format.value = 2
    end
}  

local contact_print_label_filename= dt.new_widget("label")
{
     label = _('filename without suffix'),
     ellipsize = "start",
     halign = "start"
}


local contact_print_entry_filename = dt.new_widget("entry")
{
    text = "", 
    is_password = true,
    editable = true,
    tooltip = _('enter the target image filename without suffix'),
    reset_callback = function(self) 
       self.text = "" 
    end

}

local contact_print_label_directory= dt.new_widget("label")
{
     label = _('target directory'),
     ellipsize = "start",
     halign = "start"
}

local contact_print_file_chooser_button = dt.new_widget("file_chooser_button")
{
    title = _('select the export directory'),  
    is_directory = true             
}

   

  
  
local widget = dt.new_widget("box") {
    orientation = "vertical",
    contact_print_label_options,
    contact_print_label_title,
    contact_print_entry_title,
    contact_print_combobox_images_per_line,
    contact_print_combobox_lines_per_print,
    contact_print_combobox_column_width,
    contact_print_combobox_line_width,
    contact_print_check_button_rotate,
    contact_print_check_button_border,
    contact_print_check_button_print_filename,
    contact_print_combobox_font_size,
    contact_print_separator1,
    label_target,
    contact_print_combobox_output_format,
    contact_print_label_filename,
    contact_print_entry_filename,
    contact_print_label_directory,
    contact_print_file_chooser_button,
}


local function checkIfBinExists(bin)
  local handle = io.popen("which "..bin)
  local result = handle:read()
  local ret
  handle:close()
  if (result) then
    ret = true
  else
    dt.print_error(bin.." not found")
    ret = false
  end


  return ret
end



local function show_status(storage, image, format, filename,
  number, total, high_quality, extra_data)
  dt.print(_('Export image for digital contact print ')..tostring(number).."/"..tostring(total))  
end




local function create_contact_print(storage, image_table, extra_data) --finalize
job = dt.gui.create_job(_('creating digital contact print'), true, stop_selection)
    -- create input filelist  
dt.print(_('creating digital contact print. please wait...'))
contact_print_cmd_input_files=""
for _,v in pairs(image_table) do   
     contact_print_cmd_input_files = contact_print_cmd_input_files.. "\""..v.."\" "
     if (contact_print_check_button_rotate.value) then
     cmd="exiftool \""..v.."\" -*Exif*Image*Width*"
     local fw = assert(io.popen(cmd, 'r'))
     local sw = assert(fw:read('*a'))
     fw:close()
     imagewidth=string.match (sw, "%d+")
     cmd="exiftool \""..v.."\" -*Exif*Image*Height*"
     local fh = assert(io.popen(cmd, 'r'))
     local sh = assert(fh:read('*a'))
     fh:close()
     imagehight=string.match (sh, "%d+")
         if (imagehight > imagewidth) then
            rot=dt.control.execute("mogrify -rotate -90 "..v)
            if (rot ~= 0) then
               dt.print(_("ERROR: image rotation doesn't work. for more informations see terminal output.")) 
               job.valid = false
            end    
            
         end
     
     end
end
job.percent = 0.2
dt.print_error(contact_print_cmd_input_files)

if (contact_print_cmd_input_files == "") then
    dt.print(_('ERROR: no files selected'))
    job.valid = false
   return 
end    
    
    -- check installed software
if (not (checkIfBinExists("montage"))) then
     dt.print(_('ERROR: montage not found. please install montage from ImageMagick.'))
     dt.print_error(_('montage not found. please install montage from ImageMagick.'))
     dt.control.execute("rm ~/.local/tmp/*.tif")
     job.valid = false
     return
end 
 
-- check if filename und directory is okay
  contact_print_cmd_output_path = contact_print_file_chooser_button.value
  if (contact_print_cmd_output_path == nil) then
     dt.print(_('ERROR: no target directory selected'))  
     job.valid = false
     return
  end
  job.percent = 0.3
  contact_print_contact_print_cmd_output_filename = contact_print_entry_filename.text
  if (contact_print_contact_print_cmd_output_filename == "") then
     dt.print(_('ERROR: no filename found.'))  
     job.valid = false
     return
  end 
  
  
-- create commands  

  if (contact_print_entry_title.text == "") then
    contact_print_cmd_title=""
  else
    contact_print_cmd_title="-title \""..contact_print_entry_title.text.."\""
  end    


  if (contact_print_check_button_border.value) then
     contact_print_cmd_border="-border 5  -bordercolor black"
  else 
     contact_print_cmd_border=""
  end
  
  if (contact_print_check_button_print_filename.value) then
     contact_print_cmd_print_filename="-set label '%[base]'"
  else 
     contact_print_cmd_print_filename=""
  end
  
  
  contact_print_cmd_images_per_line="-tile "..contact_print_combobox_images_per_line.value
  
  if (contact_print_combobox_lines_per_print.value == "auto") then
    contact_print_cmd_lines_per_print=""   
  else
    contact_print_cmd_lines_per_print="x"..contact_print_combobox_lines_per_print.value
  end
  
  if (contact_print_combobox_output_format.value == "TIFF") then
         contact_print_cmd_suffix_output_format="tif"
      elseif (contact_print_combobox_output_format.value == "JPEG") then
         contact_print_cmd_suffix_output_format="jpg"
      elseif (contact_print_combobox_output_format.value == "PNG") then
         contact_print_cmd_suffix_output_format="png"
      elseif (contact_print_combobox_output_format.value == "PNM") then
         contact_print_cmd_suffix_output_format="pnm"     
      elseif (contact_print_combobox_output_format.value == "PBM") then
         contact_print_cmd_suffix_output_format="pbm"   
      elseif (contact_print_combobox_output_format.value == "PPM") then
         contact_print_cmd_suffix_output_format="ppm"
      end
  
  
  contact_print_cmd_column_width=contact_print_combobox_column_width.value
  contact_print_cmd_line_width=contact_print_combobox_line_width.value
  contact_print_cmd_font_size="-pointsize "..contact_print_combobox_font_size.value
  contact_print_cmd_output_file="\""..contact_print_cmd_output_path.."/"..contact_print_contact_print_cmd_output_filename.."."..contact_print_cmd_suffix_output_format.."\""
  job.percent = 0.5  
  contact_print_montageStartCommand="montage "..contact_print_cmd_print_filename.." "..contact_print_cmd_font_size.." -geometry 100%x100%+"..contact_print_cmd_column_width.."+"..contact_print_cmd_line_width.." "..contact_print_cmd_border.." -background white "..contact_print_cmd_title.." "..contact_print_cmd_images_per_line..contact_print_cmd_lines_per_print.." "..contact_print_cmd_input_files.." "..contact_print_cmd_output_file
  
  dt.print_error(contact_print_montageStartCommand)

  resultmontage=dt.control.execute(contact_print_montageStartCommand)
    job.percent = 0.8
  if (resultmontage == 0) then
      if (dt.preferences.read("module_contact_print", "add_image_to_db", "bool")) then
         local image = dt.database.import(contact_print_cmd_output_path.."/"..contact_print_contact_print_cmd_output_filename.."."..contact_print_cmd_suffix_output_format)
      end
  dt.print(_('digital contact print successfully completed'))
  else
  dt.print(_("ERROR: montage doesn't work. for more informations see terminal output"))
  end
  removeFileCommand = "rm " ..contact_print_cmd_input_files
  dt.control.execute( removeFileCommand)

job.valid = false
end





-- limit export to jpeg (8 bit)
local function support_format(storage, format)
   fmt = string.lower(format.name)
  if string.match(fmt,"tiff") == nil then
    return false
  else
    return true
  end     
end  



-- Register
dt.register_storage("module_contact_print", _('digital contact print'), show_status, create_contact_print, support_format, nil, widget)

dt.preferences.register("module_contact_print",        
                        "add_image_to_db",                                        -- name
                        "bool",                                                   -- type
                        _('contact print: add contact print image to database'),     -- label
                        _('add the contact print image to the darktable database'),  -- tooltip
                        false)                                                    -- default           
                        
