--[[
  contact print plugin for darktable 2.2.X 

  copyright (c) 2017  Holger Klemm
  
  darktable is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.
  
  darktable is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with darktable.  If not, see <http://www.gnu.org/licenses/>.
]]

--[[
Version 2.1.1

ADDITIONAL SOFTWARE NEEDED FOR THIS SCRIPT
* montage (from ImageMagick)
* mogrify (from ImageMagick)
* exiftool

Change report:
* add process bar   
   
USAGE
* require this file from your main luarc config file.

This plugin will add a new storage option and calls hugin after export.
]]

local dt = require "darktable"
local gettext = dt.gettext

-- works only with darktable API version 4.0.0
dt.configuration.check_version(...,{4,0,0})

-- Tell gettext where to find the .mo file translating messages for a particular domain
gettext.bindtextdomain("contact_print",dt.configuration.config_dir.."/lua/")

local function _(msgid)
    return gettext.dgettext("contact_print", msgid)
end

-- command variables
cmd_title=""
cmd_output_path=""
cmd_output_filename=""
cmd_suffix_output_format=""
cmd_border=""
cmd_print_filename=""
cmd_images_per_line=""   
cmd_lines_per_print=""  
cmd_column_width=""
cmd_line_width=""
cmd_font_size=""
cmd_input_files=""
cmd_output_file=""
montageStartCommand=""

-- initialize combobox and checkbox value for default
if dt.preferences.read("contact_print",  "images_per_line", "integer") == 0 then
    -- defaults
   dt.preferences.write("contact_print", "images_per_line", "integer", 5)  --5 images 
   dt.preferences.write("contact_print", "lines_per_print", "integer", 1)  --auto lines  
   dt.preferences.write("contact_print", "column_width", "integer", 5)     --5 pixel vertical gap
   dt.preferences.write("contact_print", "line_width", "integer", 5)       --5 pixel horizontal gap
   dt.preferences.write("contact_print", "border", "bool", false)          --without border
   dt.preferences.write("contact_print", "filename", "bool", true)         --with filename
   dt.preferences.write("contact_print", "font_size", "integer", 4)        --font size 32
   dt.preferences.write("contact_print", "output_format", "integer", 2)    --JPEG ou   
end



local label_options= dt.new_widget("label")
{
     label = _('contact print options'),
     ellipsize = "end",
     halign = "end"
}


local label_title= dt.new_widget("label")
{
     label = _('title'),
     ellipsize = "start",
     halign = "start"
}


local entry_title = dt.new_widget("entry")
{
    text = "", 
    is_password = true,
    editable = true,
    tooltip = _('enter the contact print title'),
    reset_callback = function(self) 
       self.text = "" 
    end

}

combobox_images_per_line = dt.new_widget("combobox")
{
    label = _('images per line'), 
    tooltip =_(''),
    value =  dt.preferences.read("contact_print", "images_per_line", "integer"), 
    changed_callback = function(sel_images_per_line) 
    dt.preferences.write("contact_print", "images_per_line", "integer", sel_images_per_line.selected)
    end,
    "1", "2", "3","4","5","6","7","8","9","10","11","12","13","14","15","16","17","18","19","20",
    reset_callback = function(self_images_line)
       self_images_line.value = 5
    end
} 

combobox_lines_per_print = dt.new_widget("combobox")
{
    label = _('image lines'), 
    tooltip =_(''),
    value =  dt.preferences.read("contact_print", "lines_per_print", "integer"), 
    changed_callback = function(sel_lines_per_print) 
     dt.preferences.write("contact_print", "lines_per_print", "integer", sel_lines_per_print.selected)
    end,
    "auto","1", "2", "3","4","5","6","7","8","9","10","11","12","13","14","15","16","17","18","19","20",
    reset_callback = function(self_print)
       self_print.value = 1
    end
} 

combobox_column_width = dt.new_widget("combobox")
{
    label = _('vertical gap between the images'), 
    tooltip =_(''),
    value =  dt.preferences.read("contact_print", "column_width", "integer"), 
    changed_callback = function(sel_column_width) 
      dt.preferences.write("contact_print", "column_width", "integer", sel_column_width.selected)
    end,
    "1", "2", "3","4","5","6","7","8","9","10","11","12","13","14","15","16","17","18","19","20",
    
    reset_callback = function(self_grid)
       self_grid.value = 5
    end
} 

combobox_line_width = dt.new_widget("combobox")
{
    label = _('horizontal gap between the images'), 
    tooltip =_('creates a gap between '),
    value =  dt.preferences.read("contact_print", "line_width", "integer"), 
    changed_callback = function(sel_line_width) 
      dt.preferences.write("contact_print", "line_width", "integer", sel_line_width.selected)
    end,
    "1", "2", "3","4","5","6","7","8","9","10","11","12","13","14","15","16","17","18","19","20",
    
    reset_callback = function(self_grid)
       self_grid.value = 5
    end
} 


check_button_rotate = dt.new_widget("check_button")
{
    label = _('rotate portrait orientation image'), 
    value = dt.preferences.read("contact_print", "rotate", "bool"), 
    tooltip =_(''),  
    clicked_callback = function(checked_rotate)   
          if (checked_rotate.value) then
               dt.preferences.write("contact_print", "rotate", "bool", true)
          else
              dt.preferences.write("contact_print", "rotate", "bool", false)
          end
    end,
    reset_callback = function(self_rotate) 
       self_rotate.value = false
    end
    
}

check_button_border = dt.new_widget("check_button")
{
    label = _('images with black border'), 
    value = dt.preferences.read("contact_print", "border", "bool"), 
    tooltip =_(''),  
    clicked_callback = function(checked_border)   
          if (checked_border.value) then
               dt.preferences.write("contact_print", "border", "bool", true)
          else
              dt.preferences.write("contact_print", "border", "bool", false)
          end
    end,
    reset_callback = function(self_border) 
       self_border.value = false
    end
    
}



check_button_print_filename = dt.new_widget("check_button")
{
    label = _('images with filename'), 
    value = dt.preferences.read("contact_print", "filename", "bool"), 
    tooltip =_(''), 
    clicked_callback = function(checked_filename)   
          if (checked_filename.value) then
               dt.preferences.write("contact_print", "filename", "bool", true)
          else
              dt.preferences.write("contact_print", "filename", "bool", false)
          end
    end,
    reset_callback = function(self) 
       self.value = true
    end
}

combobox_font_size = dt.new_widget("combobox")
{
    label = _('filename font size'), 
    tooltip =_(''),
    value =  dt.preferences.read("contact_print", "font_size", "integer"), 
    changed_callback = function(sel_font_size) 
      dt.preferences.write("contact_print", "font_size", "integer", sel_font_size.selected)
    end,
    "14", "20", "26","32","40","48","60","72","88","96",
    
    reset_callback = function(self_font_size)
       self_font_size.value = 4
    end
} 



local separator1 = dt.new_widget("separator")
{
}

local label_target= dt.new_widget("label")
{
     label = _('target image'),
     ellipsize = "start",
     halign = "end"
}

combobox_output_format = dt.new_widget("combobox")
{
    label = _('file format'), 
    value = dt.preferences.read("contact_print", "output_format", "integer"), --2, "TIFF", "JPEG", "PNG","PNM","PBM","PGM","PPM",
    changed_callback = function(sel_output_format) 
      dt.preferences.write("contact_print", "output_format", "integer", sel_output_format.selected)
    end,
    "TIFF", "JPEG", "PNG","PNM","PBM","PPM",            
    reset_callback = function(self_output_format)
       self_output_format.value = 2
    end
}  

local label_filename= dt.new_widget("label")
{
     label = _('filename without suffix'),
     ellipsize = "start",
     halign = "start"
}


local entry_filename = dt.new_widget("entry")
{
    text = "", 
    is_password = true,
    editable = true,
    tooltip = _('enter the target image filename without suffix'),
    reset_callback = function(self) 
       self.text = "" 
    end

}

local label_directory= dt.new_widget("label")
{
     label = _('target directory'),
     ellipsize = "start",
     halign = "start"
}

local file_chooser_button = dt.new_widget("file_chooser_button")
{
    title = _('select the export directory'),  
    is_directory = true             
}

   

  
  
local widget = dt.new_widget("box") {
    orientation = "vertical",
    label_options,
    label_title,
    entry_title,
    combobox_images_per_line,
    combobox_lines_per_print,
    combobox_column_width,
    combobox_line_width,
    check_button_rotate,
    check_button_border,
    check_button_print_filename,
    combobox_font_size,
    separator1,
    label_target,
    combobox_output_format,
    label_filename,
    entry_filename,
    label_directory,
    file_chooser_button,
}


local function checkIfBinExists(bin)
  local handle = io.popen("which "..bin)
  local result = handle:read()
  local ret
  handle:close()
  if (result) then
    ret = true
  else
    dt.print_error(bin.." not found")
    ret = false
  end


  return ret
end



local function show_status(storage, image, format, filename,
  number, total, high_quality, extra_data)
  dt.print(_('Export image for digital contact print ')..tostring(number).."/"..tostring(total))  
end




local function create_contact_print(storage, image_table, extra_data) --finalize
job = dt.gui.create_job(_('creating digital contact print'), true, stop_selection)
    -- create input filelist  
dt.print(_('creating digital contact print. please wait...'))
cmd_input_files=""
for _,v in pairs(image_table) do   
     cmd_input_files = cmd_input_files.. "\""..v.."\" "
     if (check_button_rotate.value) then
     cmd="exiftool \""..v.."\" -*Exif*Image*Width*"
     local fw = assert(io.popen(cmd, 'r'))
     local sw = assert(fw:read('*a'))
     fw:close()
     imagewidth=string.match (sw, "%d+")
     cmd="exiftool \""..v.."\" -*Exif*Image*Height*"
     local fh = assert(io.popen(cmd, 'r'))
     local sh = assert(fh:read('*a'))
     fh:close()
     imagehight=string.match (sh, "%d+")
         if (imagehight > imagewidth) then
            rot=dt.control.execute("mogrify -rotate -90 "..v)
            if (rot ~= 0) then
               dt.print(_("ERROR: image rotation doesn't work. for more informations see terminal output.")) 
               job.valid = false
            end    
            
         end
     
     end
end
job.percent = 0.2
dt.print_error(cmd_input_files)

if (cmd_input_files == "") then
    dt.print(_('ERROR: no files selected'))
    job.valid = false
   return 
end    
    
    -- check installed software
if (not (checkIfBinExists("montage"))) then
     dt.print(_('ERROR: montage not found. please install montage from ImageMagick.'))
     dt.print_error(_('montage not found. please install montage from ImageMagick.'))
     dt.control.execute("rm ~/.local/tmp/*.tif")
     job.valid = false
     return
end 
 
-- check if filename und directory is okay
  cmd_output_path = file_chooser_button.value
  if (cmd_output_path == nil) then
     dt.print(_('ERROR: no target directory selected'))  
     job.valid = false
     return
  end
  job.percent = 0.3
  cmd_output_filename = entry_filename.text
  if (cmd_output_filename == "") then
     dt.print(_('ERROR: no filename found.'))  
     job.valid = false
     return
  end 
  
  
-- create commands  

  if (entry_title.text == "") then
    cmd_title=""
  else
    cmd_title="-title \""..entry_title.text.."\""
  end    


  if (check_button_border.value) then
     cmd_border="-border 5  -bordercolor black"
  else 
     cmd_border=""
  end
  
  if (check_button_print_filename.value) then
     cmd_print_filename="-set label '%[base]'"
  else 
     cmd_print_filename=""
  end
  
  
  cmd_images_per_line="-tile "..combobox_images_per_line.value
  
  if (combobox_lines_per_print.value == "auto") then
    cmd_lines_per_print=""   
  else
    cmd_lines_per_print="x"..combobox_lines_per_print.value
  end
  
  if (combobox_output_format.value == "TIFF") then
         cmd_suffix_output_format="tif"
      elseif (combobox_output_format.value == "JPEG") then
         cmd_suffix_output_format="jpg"
      elseif (combobox_output_format.value == "PNG") then
         cmd_suffix_output_format="png"
      elseif (combobox_output_format.value == "PNM") then
         cmd_suffix_output_format="pnm"     
      elseif (combobox_output_format.value == "PBM") then
         cmd_suffix_output_format="pbm"   
      elseif (combobox_output_format.value == "PPM") then
         cmd_suffix_output_format="ppm"
      end
  
  
  cmd_column_width=combobox_column_width.value
  cmd_line_width=combobox_line_width.value
  cmd_font_size="-pointsize "..combobox_font_size.value
  cmd_output_file="\""..cmd_output_path.."/"..cmd_output_filename.."."..cmd_suffix_output_format.."\""
  job.percent = 0.5  
  montageStartCommand="montage "..cmd_print_filename.." "..cmd_font_size.." -geometry 100%x100%+"..cmd_column_width.."+"..cmd_line_width.." "..cmd_border.." -background white "..cmd_title.." "..cmd_images_per_line..cmd_lines_per_print.." "..cmd_input_files.." "..cmd_output_file
  
  dt.print_error(montageStartCommand)

  resultmontage=dt.control.execute(montageStartCommand)
    job.percent = 0.8
  if (resultmontage == 0) then
      if (dt.preferences.read("module_contact_print", "add_image_to_db", "bool")) then
         local image = dt.database.import(cmd_output_path.."/"..cmd_output_filename.."."..cmd_suffix_output_format)
      end
  dt.print(_('digital contact print successfully completed'))
  else
  dt.print(_("ERROR: montage doesn't work. for more informations see terminal output"))
  end
  removeFileCommand = "rm " ..cmd_input_files
  dt.control.execute( removeFileCommand)

job.valid = false
end





-- limit export to jpeg (8 bit)
local function support_format(storage, format)
   fmt = string.lower(format.name)
  if string.match(fmt,"tiff") == nil then
    return false
  else
    return true
  end     
end  



-- Register
dt.register_storage("module_contact_print", _('digital contact print'), show_status, create_contact_print, support_format, nil, widget)

dt.preferences.register("module_contact_print",        
                        "add_image_to_db",                                        -- name
                        "bool",                                                   -- type
                        _('contact print: add contact print image to database'),     -- label
                        _('add the contact print image to the darktable database'),  -- tooltip
                        false)                                                    -- default           
                        
